"""
cPanel Account Checker API
Hanya khusus untuk check account functionality
Standalone version - tidak perlu import dari cpanel.py
"""
import requests
import re
import os
import time
import json
import random
import urllib.parse
import socket
import logging
from datetime import datetime

# Configure logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(levelname)s - %(message)s',
    handlers=[logging.StreamHandler()]
)
logger = logging.getLogger(__name__)

# Disable SSL warnings
import urllib3
urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

# ═══════════════════════════════════════════════════════════════════
#                          CONFIGURATION
# ═══════════════════════════════════════════════════════════════════

# Results folder - will be set from api_server.py via results_folder parameter
# Default to relative path (will be overridden by caller)
SCRIPT_DIR = os.path.dirname(os.path.abspath(__file__))
PARENT_DIR = os.path.dirname(SCRIPT_DIR)  # php/ directory

# Find public_html or public_htmls directory (for cPanel compatibility)
ROOT_DIR = None
parts = SCRIPT_DIR.split(os.sep)

# Check for public_html or public_htmls in path
for folder_name in ['public_htmls', 'public_html']:
    if folder_name in parts:
        idx = parts.index(folder_name)
        ROOT_DIR = os.sep.join(parts[:idx+1])
        break

# If not found, try to find by going up from current directory
if ROOT_DIR is None:
    current = SCRIPT_DIR
    max_levels = 5  # Don't go too far up
    for _ in range(max_levels):
        parent = os.path.dirname(current)
        if os.path.basename(parent) in ['public_htmls', 'public_html']:
            ROOT_DIR = parent
            break
        if parent == current:  # Reached root
            break
        current = parent

# If still not found, use parent directory (php/ -> web root)
if ROOT_DIR is None:
    ROOT_DIR = os.path.dirname(PARENT_DIR)

RESULTS_FOLDER = os.path.join(ROOT_DIR, "results")
# Only create folder if we have write permission
try:
    os.makedirs(RESULTS_FOLDER, exist_ok=True)
except (PermissionError, OSError) as e:
    # If can't create in ROOT_DIR, try creating in SCRIPT_DIR
    RESULTS_FOLDER = os.path.join(SCRIPT_DIR, "results")
    try:
        os.makedirs(RESULTS_FOLDER, exist_ok=True)
    except (PermissionError, OSError):
        # Last resort: use current directory
        RESULTS_FOLDER = os.path.join(os.getcwd(), "results")
        os.makedirs(RESULTS_FOLDER, exist_ok=True)

# ═══════════════════════════════════════════════════════════════════
#                          HELPER FUNCTIONS
# ═══════════════════════════════════════════════════════════════════

def build_proxy_url(config):
    """Build proxy URL from config"""
    proxy_type = config.get("proxy_type", "http").lower()
    proxy_host = config.get("proxy_host", "127.0.0.1:8080")
    proxy_username = config.get("proxy_username", "")
    proxy_password = config.get("proxy_password", "")
    
    if "://" in proxy_host:
        proxy_host = proxy_host.split("://")[1]
    
    if proxy_username and proxy_password:
        proxy_username_encoded = urllib.parse.quote(proxy_username)
        proxy_password_encoded = urllib.parse.quote(proxy_password)
        proxy_url = f"{proxy_type}://{proxy_username_encoded}:{proxy_password_encoded}@{proxy_host}"
    else:
        proxy_url = f"{proxy_type}://{proxy_host}"
    
    return proxy_url

def get_public_ip(use_proxy=False, proxy=None):
    """Get public IP address"""
    try:
        session = requests.Session()
        if use_proxy and proxy:
            session.proxies = {"http": proxy, "https": proxy}
        
        response = session.get("https://api.ipify.org", timeout=10)
        if response.status_code == 200:
            return ''.join(c for c in response.text if c.isprintable()).strip()
    except:
        pass
    return "Unknown"

def save_live_account(host, username, password, account_type, reseller_count, email_success, 
                     all_domains, existing_emails, successful_domains, failed_domains,
                     check_id=None, ip="Unknown", results_folder=None):
    """Save live account to file"""
    if check_id and results_folder:
        check_folder = os.path.join(results_folder, check_id)
        os.makedirs(check_folder, exist_ok=True)
        live_file = os.path.join(check_folder, "live.txt")
    else:
        live_file = "cpanel_live.txt"
    
    # Format domains
    domains_str = ','.join(all_domains) if all_domains else "None"
    
    # Format email status
    email_status = "YES" if email_success else "NO"
    
    # Get timestamp
    timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    
    # Format output: host|username|password|Type:...|Subs:0|Email:...|Domains:...|IP:xxx|timestamp
    with open(live_file, 'a', encoding='utf-8') as f:
        f.write(f"{host}|{username}|{password}|Type:{account_type}|Subs:{reseller_count}|Email:{email_status}|Domains:{domains_str}|IP:{ip}|{timestamp}\n")

def save_dead_account(host, username, password, reason, check_id=None, ip="Unknown", results_folder=None):
    """Save dead account to file"""
    timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    if check_id and results_folder:
        check_folder = os.path.join(results_folder, check_id)
        os.makedirs(check_folder, exist_ok=True)
        dead_file = os.path.join(check_folder, "dead.txt")
    else:
        dead_file = "cpanel_dead.txt"
    
    with open(dead_file, 'a', encoding='utf-8') as f:
        f.write(f"[{timestamp}] {host}|{username}|{password}|IP:{ip} → {reason}\n")

def save_error_log(host, username, password, error, check_id=None, ip="Unknown", results_folder=None):
    """Save error log to file"""
    timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    if check_id and results_folder:
        check_folder = os.path.join(results_folder, check_id)
        os.makedirs(check_folder, exist_ok=True)
        error_file = os.path.join(check_folder, "error.txt")
    else:
        error_file = "cpanel_error.txt"
    
    with open(error_file, 'a', encoding='utf-8') as f:
        f.write(f"[{timestamp}] {host}|{username}|{password}|IP:{ip} → {error}\n")

# ═══════════════════════════════════════════════════════════════════
#                          CPANEL AUTO LOGIN CLASS
# ═══════════════════════════════════════════════════════════════════

class CPanelAutoLogin:
    def __init__(self, host: str, username: str, password: str, use_proxy: bool = False, proxy_url: str = None):
        """Initialize cPanel Auto Login"""
        self.host = host
        self.username = username
        self.password = password
        self.use_proxy = use_proxy
        self.session = requests.Session()
        self.session.verify = False
        
        # Configure proxy if provided
        if use_proxy and proxy_url:
            self.session.proxies = {
                'http': proxy_url,
                'https': proxy_url
            }
            logger.info(f"[PROXY] Using proxy: {proxy_url}")
        else:
            logger.info(f"[PROXY] Proxy disabled - using direct connection")
        
        # Set headers to mimic browser
        self.session.headers.update({
            'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/137.0.0.0 Safari/537.36',
            'Accept': '*/*',
            'Accept-Language': 'en-US,en;q=0.9',
            'Accept-Encoding': 'gzip, deflate, br',
            'Connection': 'keep-alive',
            'Sec-Ch-Ua': '"Chromium";v="137", "Not/A)Brand";v="24"',
            'Sec-Ch-Ua-Mobile': '?0',
            'Sec-Ch-Ua-Platform': '"Windows"',
            'Sec-Fetch-Site': 'same-origin',
            'Sec-Fetch-Mode': 'cors',
            'Sec-Fetch-Dest': 'empty',
            'Priority': 'u=1, i'
        })
        
        # Handle host with or without port
        if ':' in self.host:
            self.base_url = f"https://{self.host}"
        else:
            # Default to port 2083 if no port specified
            self.base_url = f"https://{self.host}:2083"
        self.cpsess_id = None
    
    def get_hostname_from_ip(self, ip_address):
        """Get hostname from IP address"""
        try:
            hostname = socket.gethostbyaddr(ip_address)[0]
            return hostname
        except:
            return ip_address
    
    def is_ip_address(self, host):
        """Check if host is an IP address"""
        try:
            socket.inet_aton(host)
            return True
        except:
            return False
    
    def get_actual_hostname(self, host):
        """Get actual hostname, convert IP to hostname if needed"""
        if self.is_ip_address(host):
            hostname = self.get_hostname_from_ip(host)
            logger.info(f"[INFO] IP {host} resolved to hostname: {hostname}")
            return hostname
        return host
    
    def login(self) -> bool:
        """Perform auto-login to cPanel"""
        try:
            # Store original host for port preservation
            original_host = self.host
            # Extract hostname without port for hostname resolution
            host_only = self.host.split(':')[0] if ':' in self.host else self.host
            # Get actual hostname
            actual_host = self.get_actual_hostname(host_only)
            
            # Try login with original host first (includes port if specified)
            if self._attempt_login(self.host):
                return True
            
            # If failed and hostname is different, try with resolved hostname
            if actual_host != host_only:
                logger.info(f"[INFO] Trying login with resolved hostname: {actual_host}")
                # Preserve port from original host
                if ':' in original_host:
                    port_part = original_host.split(':')[1]
                    host_with_port = f"{actual_host}:{port_part}"
                else:
                    host_with_port = f"{actual_host}:2083"
                if self._attempt_login(host_with_port):
                    self.host = host_with_port
                    self.base_url = f"https://{host_with_port}"
                    return True
            
            return False
            
        except Exception as e:
            logger.error(f"Login error: {str(e)}")
            return False
    
    def _attempt_login(self, host):
        """Attempt login with specific host"""
        try:
            logger.info(f"Attempting to login to {host} with username: {self.username}")
            
            # Get login page
            login_url = f"https://{host}/"
            response = self.session.get(login_url, timeout=30)
            
            if response.status_code != 200:
                logger.error(f"Failed to access login page. Status: {response.status_code}")
                return False
            
            # Perform login
            login_data = {
                'user': self.username,
                'pass': self.password,
                'goto_uri': '/'
            }
            
            login_endpoint = f"https://{host}/login/?login_only=1"
            self.session.headers.update({
                'Content-Type': 'application/x-www-form-urlencoded',
                'Content-Length': str(len('&'.join([f'{k}={v}' for k, v in login_data.items()]))),
                'Origin': f"https://{host}",
                'Referer': f"https://{host}/"
            })
            
            response = self.session.post(login_endpoint, data=login_data, timeout=30)
            
            if response.status_code == 200:
                # Check if login was successful
                if 'cpsess' in response.text or 'cpsess' in str(response.cookies):
                    # Extract cpsess ID
                    cpsess_match = re.search(r'cpsess(\d+)', response.text)
                    if cpsess_match:
                        self.cpsess_id = cpsess_match.group(0)
                        logger.info(f"Login successful! Session ID: {self.cpsess_id}")
                        return True
                    else:
                        # Try to extract from cookies
                        for cookie in self.session.cookies:
                            if 'cpsess' in cookie.name:
                                self.cpsess_id = cookie.name
                                logger.info(f"Login successful! Session ID: {self.cpsess_id}")
                                return True
                
                # Check for error messages
                if 'error' in response.text.lower() or 'invalid' in response.text.lower():
                    logger.error("Login failed: Invalid credentials or error message found")
                    return False
                
                logger.info("Login appears successful (no explicit error found)")
                return True
            else:
                logger.error(f"Login failed with status code: {response.status_code}")
                return False
                
        except Exception as e:
            logger.error(f"Login attempt failed for {host}: {str(e)}")
            return False
    
    def get_domains_and_zones(self, debug: bool = False):
        """Get list of domains and DNS zones using DomainLookup API"""
        try:
            domains = set()
            dns_zones = set()
            
            # Get domains from cPanel using DomainInfo API
            # Use execute API with json parameter
            domains_url = f"{self.base_url}/{self.cpsess_id}/execute/DomainInfo/domains_data"
            domains_data = {
                'api.version': 1
            }
            
            # Add json parameter to ensure JSON response
            domains_url_with_json = f"{domains_url}?json=1"
            
            # Set headers for JSON response
            headers = {
                'Accept': 'application/json, */*',
                'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8',
                'X-Requested-With': 'XMLHttpRequest'
            }
            
            response = self.session.post(domains_url_with_json, data=domains_data, headers=headers, verify=False, timeout=30)
            
            if debug:
                logger.info(f"[DEBUG] Domains API response status: {response.status_code}")
            
            if response.status_code == 200:
                try:
                    result = response.json()
                    if debug:
                        logger.info(f"[DEBUG] Domains API response: {result}")
                    
                    if 'data' in result and isinstance(result['data'], list):
                        for domain_info in result['data']:
                            if isinstance(domain_info, dict):
                                if 'domain' in domain_info:
                                    domains.add(domain_info['domain'])
                                if 'subdomain' in domain_info:
                                    domains.add(domain_info['subdomain'])
                    elif 'data' in result and isinstance(result['data'], dict):
                        # Handle case where data is a dict instead of list
                        for key, value in result['data'].items():
                            if isinstance(value, dict) and 'domain' in value:
                                domains.add(value['domain'])
                except json.JSONDecodeError:
                    logger.error(f"Failed to parse domains response as JSON: {response.text}")
                except Exception as e:
                    logger.error(f"Error processing domains data: {str(e)}")
                    if debug:
                        logger.info(f"[DEBUG] Raw domains response: {response.text}")
            
            # Get DNS zones using DomainLookup API (new method)
            # Use json-api endpoint with json parameter
            dns_url = f"{self.base_url}/{self.cpsess_id}/json-api/cpanel"
            dns_data = {
                'cpanel_jsonapi_apiversion': '2',
                'cpanel_jsonapi_module': 'DomainLookup',
                'cpanel_jsonapi_func': 'getbasedomains',
                'cpanel_jsonapi_user': self.username
            }
            
            # Add json parameter to URL to force JSON mode
            dns_url_with_json = f"{dns_url}?json=1"
            
            # Set headers for JSON API
            headers = {
                'Accept': 'application/json, */*',
                'Accept-Language': 'en-US,en;q=0.9,id;q=0.8',
                'Connection': 'keep-alive',
                'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8',
                'Origin': self.base_url,
                'Referer': f"{self.base_url}/",
                'Sec-Fetch-Dest': 'empty',
                'Sec-Fetch-Mode': 'cors',
                'Sec-Fetch-Site': 'same-origin',
                'User-Agent': 'Mozilla/5.0 (iPhone; CPU iPhone OS 18_5 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/18.5 Mobile/15E148 Safari/604.1',
                'X-Requested-With': 'XMLHttpRequest'
            }
            
            response = self.session.post(dns_url_with_json, data=dns_data, headers=headers, verify=False, timeout=30)
            
            if debug:
                logger.info(f"[DEBUG] DNS Zone API response status: {response.status_code}")
            
            if response.status_code == 200:
                try:
                    result = response.json()
                    if debug:
                        logger.info(f"[DEBUG] DNS Zone API response: {result}")
                    
                    # Parse the JSON API response
                    if 'cpanelresult' in result:
                        cpanel_result = result['cpanelresult']
                        if 'data' in cpanel_result and isinstance(cpanel_result['data'], list):
                            for dns_info in cpanel_result['data']:
                                if isinstance(dns_info, dict):
                                    # Look for domain names in various fields
                                    for field in ['domain', 'name', 'basedomain']:
                                        if field in dns_info and dns_info[field]:
                                            dns_zones.add(dns_info[field])
                        elif 'data' in cpanel_result and isinstance(cpanel_result['data'], dict):
                            # Handle case where data is a dict
                            for key, value in cpanel_result['data'].items():
                                if isinstance(value, dict):
                                    for field in ['domain', 'name', 'basedomain']:
                                        if field in value and value[field]:
                                            dns_zones.add(value[field])
                                elif isinstance(value, str) and value:
                                    dns_zones.add(value)
                except json.JSONDecodeError:
                    logger.error(f"Failed to parse DNS zone response as JSON: {response.text}")
                except Exception as e:
                    logger.error(f"Error processing DNS zone data: {str(e)}")
                    if debug:
                        logger.info(f"[DEBUG] Raw DNS zone response: {response.text}")
            else:
                logger.warning(f"DNS Zone API failed with status: {response.status_code}")
            
            # Combine and remove duplicates
            all_domains = domains.union(dns_zones)
            
            if debug:
                logger.info(f"[DEBUG] Found domains: {list(domains)}")
                logger.info(f"[DEBUG] Found DNS zones: {list(dns_zones)}")
                logger.info(f"[DEBUG] Combined unique domains: {list(all_domains)}")
            
            return list(all_domains)
            
        except Exception as e:
            logger.error(f"Error getting domains and zones: {str(e)}")
            return []

    def get_email_accounts(self, debug: bool = False) -> list:
        """Get list of email accounts using list_pops_with_disk API"""
        try:
            email_url = f"{self.base_url}/{self.cpsess_id}/execute/Email/list_pops_with_disk"
            email_data = {
                'api.sort': '1',
                'api.sort_column_0': 'user',
                'api.sort_method_0': 'lexicographic',
                'api.sort_reverse_0': '0',
                'api.paginate': '1',
                'api.paginate_start': '1',
                'api.paginate_size': '20',
                'api.paginate_page': '1',
                'no_human_readable_keys': '1',
                'get_restrictions': '1',
                'include_main': '1'
            }
            
            # Add json parameter to ensure JSON response
            email_url_with_json = f"{email_url}?json=1"
            
            # Set headers for email list API
            headers = {
                'Accept': 'application/json, */*',
                'Accept-Language': 'en-US,en;q=0.9,id;q=0.8',
                'Connection': 'keep-alive',
                'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8',
                'Origin': self.base_url,
                'Referer': f"{self.base_url}/",
                'Sec-Fetch-Dest': 'empty',
                'Sec-Fetch-Mode': 'cors',
                'Sec-Fetch-Site': 'same-origin',
                'User-Agent': 'Mozilla/5.0 (Linux; Android 6.0; Nexus 5 Build/MRA58N) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/140.0.0.0 Mobile Safari/537.36',
                'X-Requested-With': 'XMLHttpRequest',
                'sec-ch-ua': '"Chromium";v="140", "Not=A?Brand";v="24", "Google Chrome";v="140"',
                'sec-ch-ua-mobile': '?1',
                'sec-ch-ua-platform': '"Android"'
            }
            
            response = self.session.post(email_url_with_json, data=email_data, headers=headers, verify=False, timeout=30)
            
            if debug:
                logger.info(f"[DEBUG] Email list API response status: {response.status_code}")
            
            email_accounts = []
            if response.status_code == 200:
                try:
                    result = response.json()
                    if debug:
                        logger.info(f"[DEBUG] Email list API response: {result}")
                    
                    if 'data' in result and isinstance(result['data'], list):
                        for email_info in result['data']:
                            if isinstance(email_info, dict) and 'user' in email_info:
                                email_accounts.append(email_info['user'])
                except json.JSONDecodeError:
                    logger.error(f"Failed to parse email list response as JSON: {response.text}")
                except Exception as e:
                    logger.error(f"Error processing email list data: {str(e)}")
                    if debug:
                        logger.info(f"[DEBUG] Raw email list response: {response.text}")
            
            if debug:
                logger.info(f"[DEBUG] Found email accounts: {email_accounts}")
            
            return email_accounts
            
        except Exception as e:
            logger.error(f"Error getting email accounts: {str(e)}")
            return []

    def check_reseller_status_new(self, debug: bool = False) -> dict:
        """Check reseller status using Resellers/list_accounts API"""
        try:
            reseller_url = f"{self.base_url}/{self.cpsess_id}/execute/Resellers/list_accounts"
            
            # Add json parameter to ensure JSON response
            reseller_url_with_json = f"{reseller_url}?json=1"
            
            # Set headers for reseller API
            headers = {
                'Accept': 'application/json, */*',
                'Accept-Language': 'en-US,en;q=0.9,id;q=0.8',
                'Connection': 'keep-alive',
                'Content-Length': '0',
                'Origin': self.base_url,
                'Referer': f"{self.base_url}/",
                'Sec-Fetch-Dest': 'empty',
                'Sec-Fetch-Mode': 'cors',
                'Sec-Fetch-Site': 'same-origin',
                'User-Agent': 'Mozilla/5.0 (iPhone; CPU iPhone OS 18_5 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/18.5 Mobile/15E148 Safari/604.1',
                'X-Requested-With': 'XMLHttpRequest'
            }
            
            response = self.session.post(reseller_url_with_json, headers=headers, verify=False, timeout=30)
            
            if debug:
                logger.info(f"[DEBUG] Reseller API response status: {response.status_code}")
            
            is_reseller = False
            reseller_accounts = []
            
            if response.status_code == 200:
                try:
                    result = response.json()
                    if debug:
                        logger.info(f"[DEBUG] Reseller API response: {result}")
                    
                    if 'data' in result and isinstance(result['data'], list) and len(result['data']) > 0:
                        data_list = result['data']
                        
                        if len(data_list) == 1:
                            first_account = data_list[0]
                            if isinstance(first_account, dict) and first_account.get('select') == '1':
                                is_reseller = False
                            else:
                                is_reseller = True
                                reseller_accounts = data_list
                        else:
                            is_reseller = True
                            reseller_accounts = data_list
                except json.JSONDecodeError:
                    logger.error(f"Failed to parse reseller response as JSON: {response.text}")
                except Exception as e:
                    logger.error(f"Error processing reseller data: {str(e)}")
                    if debug:
                        logger.info(f"[DEBUG] Raw reseller response: {response.text}")
            else:
                logger.warning(f"Reseller API failed with status: {response.status_code}")
            
            return {
                'is_reseller': is_reseller,
                'accounts': reseller_accounts,
                'account_count': len(reseller_accounts),
                'current_username_in_list': False
            }
            
        except Exception as e:
            logger.error(f"Error checking reseller status: {str(e)}")
            return {'is_reseller': False, 'accounts': [], 'account_count': 0}

    def check_email_settings(self, email_account: str, test_email: str, debug: bool = False) -> dict:
        """Check email settings and send test email for domains that haven't been tested"""
        try:
            if not self.cpsess_id:
                logger.error("Not logged in. Please login first.")
                return {'success': False, 'error': 'Not logged in'}
            
            logger.info(f"Checking email settings for account: {email_account}")
            
            # Get all domains and zones
            all_domains = self.get_domains_and_zones(debug)
            
            # Get existing email accounts to avoid testing domains that already have email
            existing_email_accounts = self.get_email_accounts(debug)
            
            # Filter domains that don't have email accounts yet
            untested_domains = []
            for domain in all_domains:
                domain_has_email = False
                for existing_email in existing_email_accounts:
                    if domain in existing_email or existing_email in domain:
                        domain_has_email = True
                        break
                
                if not domain_has_email:
                    untested_domains.append(domain)
            
            if debug:
                logger.info(f"[DEBUG] All domains: {all_domains}")
                logger.info(f"[DEBUG] Existing email accounts: {existing_email_accounts}")
                logger.info(f"[DEBUG] Untested domains (no email accounts): {untested_domains}")
            
            # Test email only for untested domains
            email_results = {}
            overall_success = False
            
            for domain in untested_domains:
                try:
                    # Prepare email dispatch data for this domain
                    email_data = {
                        'to': test_email,
                        'account': email_account,
                        'block_on_send': 'true',
                        'domain': domain
                    }
                    
                    # Update headers for email dispatch
                    headers = {
                        'Accept': 'application/json, */*',
                        'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8',
                        'Content-Length': str(len('&'.join([f'{k}={v}' for k, v in email_data.items()]))),
                        'X-Requested-With': 'XMLHttpRequest'
                    }
                    self.session.headers.update(headers)
                    
                    # Send email dispatch request
                    # Add json parameter to ensure JSON response
                    email_endpoint = f"{self.base_url}/{self.cpsess_id}/execute/Email/dispatch_client_settings"
                    email_endpoint_with_json = f"{email_endpoint}?json=1"
                    response = self.session.post(email_endpoint_with_json, data=email_data, timeout=30)
                    
                    domain_result = {
                        'success': False,
                        'status_code': response.status_code,
                        'response': response.text,
                        'timestamp': time.time(),
                        'domain': domain
                    }
                    
                    if response.status_code == 200:
                        try:
                            json_response = response.json()
                            domain_result.update(json_response)
                            domain_result['success'] = True
                            overall_success = True
                        except json.JSONDecodeError:
                            if 'success' in response.text.lower() or 'sent' in response.text.lower():
                                domain_result['success'] = True
                                overall_success = True
                    
                    email_results[domain] = domain_result
                    
                except Exception as e:
                    logger.error(f"[ERROR] Email check error for {domain}: {str(e)}")
                    email_results[domain] = {
                        'success': False,
                        'error': str(e),
                        'domain': domain,
                        'timestamp': time.time()
                    }
            
            # Create combined result
            result = {
                'success': overall_success,
                'all_domains': all_domains,
                'existing_email_accounts': existing_email_accounts,
                'untested_domains': untested_domains,
                'domains_tested': len(untested_domains),
                'successful_domains': [d for d, r in email_results.items() if r.get('success', False)],
                'failed_domains': [d for d, r in email_results.items() if not r.get('success', False)],
                'domain_results': email_results,
                'timestamp': time.time()
            }
            
            return result
            
        except Exception as e:
            logger.error(f"Email check error: {str(e)}")
            return {'success': False, 'error': str(e)}
    
    def logout(self):
        """Logout from cPanel"""
        try:
            if self.cpsess_id:
                logout_url = f"{self.base_url}/{self.cpsess_id}/logout"
                self.session.get(logout_url, timeout=10)
                logger.info("Logged out successfully")
        except Exception as e:
            logger.error(f"Logout error: {str(e)}")
        finally:
            self.session.close()

# ═══════════════════════════════════════════════════════════════════
#                          ACCOUNT CHECKER
# ═══════════════════════════════════════════════════════════════════

def check_account(host, username, password, use_proxy=False, proxy=None, 
                  callback_success=None, callback_error=None, callback_progress=None,
                  check_id=None, results_folder=None, test_email="test@example.com"):
    """
    Check cPanel account
    
    Args:
        host: cPanel host (domain or IP, with or without port)
        username: cPanel username
        password: cPanel password
        use_proxy: Use proxy or not
        proxy: Proxy URL
        callback_success: Callback function for success
        callback_error: Callback function for error
        callback_progress: Callback function for progress
        check_id: Check ID for web mode
        results_folder: Results folder path
        test_email: Test email for email dispatch check
    
    Returns:
        dict: Result with status, message, and data
    """
    try:
        # Clean host - handle port format (host:port or just host)
        # cPanel typically uses port 2083 for HTTPS
        original_host = host
        host_with_port = host
        if ':' in host:
            host_parts = host.split(':')
            host_only = host_parts[0]
            port = host_parts[1] if len(host_parts) > 1 else '2083'
            host_with_port = f"{host_only}:{port}"
        else:
            # Default to port 2083 if no port specified
            port = '2083'
            host_with_port = f"{host}:{port}"
        
        # Get public IP
        public_ip = get_public_ip(use_proxy, proxy)
        
        # Call progress callback
        if callback_progress:
            callback_progress(f"{username}@{host_with_port}", public_ip)
        
        # Build proxy URL if needed
        proxy_url = None
        if use_proxy and proxy:
            proxy_url = proxy
        
        # Create cPanel instance with host:port
        cpanel = CPanelAutoLogin(host_with_port, username, password, use_proxy=use_proxy, proxy_url=proxy_url)
        
        # Attempt login
        if cpanel.login():
            # Check reseller status
            reseller_info = cpanel.check_reseller_status_new(debug=False)
            is_reseller = reseller_info.get('is_reseller', False)
            reseller_count = reseller_info.get('account_count', 0)
            
            account_type = "RESELLER" if is_reseller else "REGULAR"
            
            # Use username as email account
            email_account = username
            
            # Check email settings
            email_result = cpanel.check_email_settings(email_account, test_email, debug=False)
            email_success = email_result.get('success', False)
            
            all_domains = email_result.get('all_domains', [])
            existing_emails = email_result.get('existing_email_accounts', [])
            successful_domains = email_result.get('successful_domains', [])
            failed_domains = email_result.get('failed_domains', [])
            
            # Call success callback
            if callback_success:
                callback_success(
                    host, username, password, account_type, reseller_count, 
                    email_success, all_domains, existing_emails, 
                    successful_domains, failed_domains, public_ip
                )
            
            # Save live account
            save_live_account(
                host, username, password, account_type, reseller_count, email_success,
                all_domains, existing_emails, successful_domains, failed_domains,
                check_id, public_ip, results_folder
            )
            
            # Logout
            cpanel.logout()
            
            return {
                'status': 'success',
                'account_type': account_type,
                'reseller_count': reseller_count,
                'email_success': email_success,
                'domains': all_domains,
                'emails': existing_emails,
                'successful_domains': successful_domains,
                'failed_domains': failed_domains,
                'ip': public_ip
            }
        else:
            # Login failed - this is a dead account, not an error
            error_msg = "Login failed - Invalid credentials"
            is_dead = True  # Login failed means dead account
            if callback_error:
                callback_error(host, username, password, error_msg, public_ip, is_dead)
            save_dead_account(host, username, password, error_msg, check_id, public_ip, results_folder)
            
            # Logout if session exists
            try:
                cpanel.logout()
            except:
                pass
            
            return {'status': 'failed', 'message': error_msg, 'ip': public_ip, 'is_dead': is_dead}
            
    except Exception as e:
        error_msg = f"Error: {str(e)}"
        public_ip = get_public_ip(use_proxy, proxy) if 'public_ip' not in locals() else "Unknown"
        if callback_error:
            callback_error(host, username, password, error_msg, public_ip, False)
        save_error_log(host, username, password, error_msg, check_id, public_ip, results_folder)
        return {'status': 'failed', 'message': error_msg, 'ip': public_ip}
