"""
Python API Server for Microsoft Account Checker & cPanel Checker
This API is used by PHP web interface
"""
from flask import Flask, request, jsonify
import json
import uuid
import os
import sys
import shutil
import logging
from concurrent.futures import ThreadPoolExecutor, as_completed
import func

# Setup logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

# Import api-microsoft.py (note: Python module name with dash needs special handling)
import importlib.util
SCRIPT_DIR = os.path.dirname(os.path.abspath(__file__))
spec = importlib.util.spec_from_file_location("api_microsoft", os.path.join(SCRIPT_DIR, "api-microsoft.py"))
api_microsoft = importlib.util.module_from_spec(spec)
spec.loader.exec_module(api_microsoft)

# Import api-cpanel.py
spec_cpanel = importlib.util.spec_from_file_location("api_cpanel", os.path.join(SCRIPT_DIR, "api-cpanel.py"))
api_cpanel = importlib.util.module_from_spec(spec_cpanel)
spec_cpanel.loader.exec_module(api_cpanel)

app = Flask(__name__)

# CORS headers manually (in case flask_cors is not installed)
@app.after_request
def after_request(response):
    response.headers.add('Access-Control-Allow-Origin', '*')
    response.headers.add('Access-Control-Allow-Headers', 'Content-Type')
    response.headers.add('Access-Control-Allow-Methods', 'GET, POST, OPTIONS')
    return response

# Health check endpoint
@app.route('/api/health', methods=['GET'])
def health_check():
    """Health check endpoint for connection testing"""
    return jsonify({
        'status': 'ok',
        'message': 'Python API Server is running',
        'version': '1.0'
    })

# Global state
check_status = {}  # {check_id: {status, progress, total, ...}}
check_results = {}  # {check_id: {results: [], ...}}
running_threads = {}  # {check_id: thread}

# Results folder - relative to public_html directory
SCRIPT_DIR = os.path.dirname(os.path.abspath(__file__))
PARENT_DIR = os.path.dirname(SCRIPT_DIR)  # php/ directory

# Find public_html or public_htmls directory (for cPanel compatibility)
# Use the same logic as func.py
ROOT_DIR = None
parts = SCRIPT_DIR.split(os.sep)

# Check for public_html or public_htmls in path
for folder_name in ['public_htmls', 'public_html']:
    if folder_name in parts:
        idx = parts.index(folder_name)
        ROOT_DIR = os.sep.join(parts[:idx+1])
        break

# If not found, try to find by going up from current directory
if ROOT_DIR is None:
    current = SCRIPT_DIR
    max_levels = 5  # Don't go too far up
    for _ in range(max_levels):
        parent = os.path.dirname(current)
        if os.path.basename(parent) in ['public_htmls', 'public_html']:
            ROOT_DIR = parent
            break
        if parent == current:  # Reached root
            break
        current = parent

# If still not found, use parent directory (php/ -> web root)
if ROOT_DIR is None:
    ROOT_DIR = os.path.dirname(PARENT_DIR)

RESULTS_FOLDER = os.path.join(ROOT_DIR, "results")
# Only create folder if we have write permission
try:
    os.makedirs(RESULTS_FOLDER, exist_ok=True)
except (PermissionError, OSError) as e:
    # If can't create in ROOT_DIR, try creating in SCRIPT_DIR
    RESULTS_FOLDER = os.path.join(SCRIPT_DIR, "results")
    try:
        os.makedirs(RESULTS_FOLDER, exist_ok=True)
    except (PermissionError, OSError):
        # Last resort: use current directory
        RESULTS_FOLDER = os.path.join(os.getcwd(), "results")
        os.makedirs(RESULTS_FOLDER, exist_ok=True)

# Restore all checks from user data on startup
def restore_all_checks():
    """Restore all checks (completed, stopped, running) from user data"""
    func.load_user_data()
    restored_count = 0
    for username, user_info in func.user_data_cache.items():
        user_checks = user_info.get('checks', {})
        for check_id, check_info in user_checks.items():
            if check_id not in check_status:
                restored_status = {
                    'status': check_info.get('status', 'completed'),
                    'progress': check_info.get('progress', 0),
                    'total_accounts': check_info.get('total_accounts', 0),
                    'completed_accounts': check_info.get('completed_accounts', 0),
                    'success': check_info.get('success', 0),
                    'failed': check_info.get('failed', 0),
                    'start_time': check_info.get('start_time'),
                    'username': username,
                    'checking_ip': check_info.get('checking_ip', 'Unknown'),
                    'config': check_info.get('config')
                }
                check_status[check_id] = restored_status
                restored_count += 1
    return restored_count

# Restore checks on startup
restore_all_checks()

# ═══════════════════════════════════════════════════════════════════
#                          API ENDPOINTS
# ═══════════════════════════════════════════════════════════════════

@app.route('/api/check/start', methods=['POST'])
def api_check_start():
    """Start checking accounts"""
    try:
        data = request.json
        accounts = data.get('accounts', [])  # List of [email, password]
        config = data.get('config', {})  # {use_proxy, proxy_type, proxy_host, proxy_username, proxy_password, threads}
        username = data.get('username', 'admin')
        
        # If config is not provided or empty, load from user's saved config
        if not config:
            config = func.load_user_config(username)
        
        # Ensure threads is an integer and within valid range
        if 'threads' in config:
            try:
                config['threads'] = int(config['threads'])
            except (ValueError, TypeError):
                config['threads'] = 5
            config['threads'] = max(1, min(50, config['threads']))
        else:
            config['threads'] = 5
        
        if not accounts:
            return jsonify({'error': 'No accounts provided'}), 400
        
        # Generate check ID
        check_id = str(uuid.uuid4())
        
        # Initialize check status
        from datetime import datetime
        start_time = datetime.now().isoformat()
        check_status[check_id] = {
            'status': 'running',
            'progress': 0,
            'total_accounts': len(accounts),
            'completed_accounts': 0,
            'success': 0,
            'failed': 0,
            'start_time': start_time,
            'username': username,
            'checking_ip': 'Unknown',
            'config': config,  # Save config used for this check
            'service': config.get('service', 'Microsoft Account Checker')  # Save service name
        }
        check_results[check_id] = {
            'results': [],
            'start_time': None
        }
        
        # Start checking in background thread
        thread = ThreadPoolExecutor(max_workers=1)
        future = thread.submit(run_checker, check_id, accounts, config, username)
        running_threads[check_id] = {'thread': thread, 'future': future}
        
        # Send notification if enabled
        service_name = config.get('service', 'Microsoft Account Checker')
        notification_msg = f"""🔔 <b>Check Started</b>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

✅ <b>Your check has been started!</b>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

📋 <b>Check Details:</b>

🔹 <b>Service:</b> {service_name}
🔹 <b>Total Accounts:</b> {len(accounts)}
🔹 <b>Check ID:</b>
<code>{check_id}</code>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

⏳ <b>Status:</b> Running
📊 <b>Progress:</b> 0%

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

💡 You will be notified when the check is completed.

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"""
        func.send_user_notification(username, notification_msg)
        
        return jsonify({
            'success': True,
            'check_id': check_id,
            'message': 'Check started'
        })
    except Exception as e:
        return jsonify({'error': str(e)}), 500

@app.route('/api/check/status/<check_id>', methods=['GET'])
def api_check_status(check_id):
    """Get checking status"""
    # Check in memory first
    if check_id in check_status:
        status = check_status[check_id].copy()
        return jsonify(status)
    
    # If not in memory, try to restore from user data
    func.load_user_data()
    for username, user_info in func.user_data_cache.items():
        user_checks = user_info.get('checks', {})
        if check_id in user_checks:
            check_info = user_checks[check_id]
            # Restore to memory
            restored_status = {
                'status': check_info.get('status', 'completed'),
                'progress': check_info.get('progress', 0),
                'total_accounts': check_info.get('total_accounts', 0),
                'completed_accounts': check_info.get('completed_accounts', 0),
                'success': check_info.get('success', 0),
                'failed': check_info.get('failed', 0),
                'start_time': check_info.get('start_time'),
                'username': username,
                'checking_ip': check_info.get('checking_ip', 'Unknown'),
                'config': check_info.get('config')
            }
            check_status[check_id] = restored_status
            return jsonify(restored_status)
    
    return jsonify({'error': 'Check ID not found'}), 404

@app.route('/api/check/stop/<check_id>', methods=['POST'])
def api_check_stop(check_id):
    """Stop a running check"""
    if check_id not in check_status:
        return jsonify({'error': 'Check ID not found'}), 404
    
    check_status[check_id]['status'] = 'stopped'
    
    # Cancel thread if running
    if check_id in running_threads:
        running_threads[check_id]['future'].cancel()
        del running_threads[check_id]
    
    return jsonify({'success': True, 'message': 'Check stopped'})

@app.route('/api/check/results/<check_id>', methods=['GET'])
def api_check_results(check_id):
    """Get checking results"""
    if check_id not in check_results:
        return jsonify({'error': 'Check ID not found'}), 404
    
    return jsonify(check_results[check_id])

@app.route('/api/user/login', methods=['POST'])
def api_user_login():
    """User login - step 1 (supports username or access_key)"""
    try:
        data = request.json
        username_or_key = data.get('username_or_key', '').strip() or data.get('access_key', '').strip() or data.get('username', '').strip()
        
        if not username_or_key:
            return jsonify({'success': False, 'error': 'Username or access key is required'}), 400
        
        result = func.login_user_step1(username_or_key)
        return jsonify(result)
    except Exception as e:
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/api/user/login/verify', methods=['POST'])
def api_user_login_verify():
    """User login - step 2 (OTP verification)"""
    try:
        data = request.json
        username = data.get('username', '').strip()
        access_key = data.get('access_key', '').strip()
        otp_code = data.get('otp_code', '').strip()
        
        result = func.login_user_step2(username, access_key, otp_code)
        return jsonify(result)
    except Exception as e:
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/api/user/register', methods=['POST'])
def api_user_register():
    """User registration (legacy - now handled via Telegram bot)"""
    try:
        data = request.json
        chat_id = data.get('chat_id', '').strip()
        
        # Check if user already registered
        func.load_user_data()
        for username, user_info in func.user_data_cache.items():
            if user_info.get('chat_id') == str(chat_id):
                return jsonify({
                    'success': False,
                    'error': 'You are already registered! Please use the credentials that were sent previously or register via Telegram bot with /start'
                })
        
        # Return message to use Telegram bot
        return jsonify({
            'success': False,
            'error': 'Registration must be done via Telegram bot. Please send /start to @forbidden_tools_bot to start registration.'
        })
    except Exception as e:
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/api/user/info', methods=['GET'])
def api_user_info():
    """Get user information"""
    try:
        username = request.args.get('username', 'admin')
        user_info = func.get_user(username)
        
        if not user_info:
            return jsonify({'error': 'User not found'}), 404
        
        return jsonify({
            'username': username,
            'name': user_info.get('name', 'N/A'),
            'phone': user_info.get('phone', 'N/A'),
            'telegram_username': user_info.get('telegram_username', 'N/A'),
            'ip_address': user_info.get('ip_address', 'Unknown'),
            'last_used': user_info.get('last_used', 'N/A'),
            'telegram_notification': user_info.get('telegram_notification', False),
            'chat_id': user_info.get('chat_id', 'N/A'),
            'is_admin': user_info.get('is_admin', False),
            'created_at': user_info.get('created_at', 'N/A'),
            'total_checks': user_info.get('total_checks', 0)
        })
    except Exception as e:
        return jsonify({'error': str(e)}), 500

@app.route('/api/user/by_chat_id', methods=['GET'])
def api_user_by_chat_id():
    """Get user information by chat_id (for Mini App)"""
    try:
        chat_id = request.args.get('chat_id', '')
        if not chat_id:
            return jsonify({'error': 'Chat ID is required'}), 400
        
        func.load_user_data()
        
        # Normalize chat_id to string for comparison
        chat_id_str = str(chat_id).strip()
        
        # Find user by chat_id
        # Try both string and integer comparison
        for username, user_info in func.user_data_cache.items():
            saved_chat_id = user_info.get('chat_id')
            if not saved_chat_id:
                continue
            
            # Convert to string for comparison
            saved_chat_id_str = str(saved_chat_id).strip()
            
            # Try exact match
            if saved_chat_id_str == chat_id_str:
                return jsonify({
                    'username': username,
                    'name': user_info.get('name', 'N/A'),
                    'phone': user_info.get('phone', 'N/A'),
                    'telegram_username': user_info.get('telegram_username', 'N/A'),
                    'ip_address': user_info.get('ip_address', 'Unknown'),
                    'last_used': user_info.get('last_used', 'N/A'),
                    'telegram_notification': user_info.get('telegram_notification', False),
                    'chat_id': user_info.get('chat_id', 'N/A'),
                    'is_admin': user_info.get('is_admin', False),
                    'created_at': user_info.get('created_at', 'N/A'),
                    'total_checks': user_info.get('total_checks', 0)
                })
            
            # Try integer comparison (in case one is int and other is string)
            try:
                if int(saved_chat_id_str) == int(chat_id_str):
                    return jsonify({
                        'username': username,
                        'name': user_info.get('name', 'N/A'),
                        'phone': user_info.get('phone', 'N/A'),
                        'telegram_username': user_info.get('telegram_username', 'N/A'),
                        'ip_address': user_info.get('ip_address', 'Unknown'),
                        'last_used': user_info.get('last_used', 'N/A'),
                        'telegram_notification': user_info.get('telegram_notification', False),
                        'chat_id': user_info.get('chat_id', 'N/A'),
                        'is_admin': user_info.get('is_admin', False),
                        'created_at': user_info.get('created_at', 'N/A'),
                        'total_checks': user_info.get('total_checks', 0)
                    })
            except (ValueError, TypeError):
                pass
        
        return jsonify({'error': 'User not found'}), 404
    except Exception as e:
        import traceback
        traceback.print_exc()
        return jsonify({'error': str(e)}), 500

@app.route('/api/user/notification', methods=['POST'])
def api_user_notification():
    """Update Telegram notification setting"""
    try:
        data = request.json or {}
        username = data.get('username', 'admin')
        enabled = data.get('enabled', False)
        
        user_info = func.get_user(username)
        if not user_info:
            return jsonify({'success': False, 'error': 'User not found'}), 404
        
        user_info['telegram_notification'] = bool(enabled)
        if func.save_user_data(username, user_info):
            return jsonify({'success': True, 'message': 'Notification setting updated'})
        else:
            return jsonify({'success': False, 'error': 'Failed to save setting'}), 500
    except Exception as e:
        return jsonify({'success': False, 'error': str(e)}), 500

# Admin endpoints
@app.route('/api/admin/users', methods=['GET'])
def api_admin_users():
    """Get all users (admin only)"""
    try:
        admin_username = request.args.get('admin_username', 'admin')
        admin_user = func.get_user(admin_username)
        
        if not admin_user or not admin_user.get('is_admin', False):
            return jsonify({'error': 'Admin access required'}), 403
        
        func.load_user_data()
        users = []
        for username, user_info in func.user_data_cache.items():
            users.append({
                'username': username,
                'name': user_info.get('name', 'N/A'),
                'phone': user_info.get('phone', 'N/A'),
                'telegram_username': user_info.get('telegram_username', 'N/A'),
                'chat_id': user_info.get('chat_id', 'N/A'),
                'is_admin': user_info.get('is_admin', False),
                'created_at': user_info.get('created_at', 'N/A'),
                'last_used': user_info.get('last_used', 'N/A'),
                'total_checks': user_info.get('total_checks', 0)
            })
        
        return jsonify({'success': True, 'users': users})
    except Exception as e:
        return jsonify({'error': str(e)}), 500

@app.route('/api/admin/user/add', methods=['POST'])
def api_admin_add_user():
    """Add new user (admin only)"""
    try:
        data = request.json or {}
        admin_username = data.get('admin_username', 'admin')
        admin_user = func.get_user(admin_username)
        
        if not admin_user or not admin_user.get('is_admin', False):
            return jsonify({'success': False, 'error': 'Admin access required'}), 403
        
        username = data.get('username', '').strip()
        name = data.get('name', '').strip()
        phone = data.get('phone', '').strip()
        access_key = data.get('access_key', '').strip()
        chat_id = data.get('chat_id', '').strip()
        telegram_username = data.get('telegram_username', '').strip()
        is_admin = data.get('is_admin', False)
        
        if not username or not access_key:
            return jsonify({'success': False, 'error': 'Username and access key are required'}), 400
        
        if func.user_exists(username):
            return jsonify({'success': False, 'error': 'Username already exists'}), 400
        
        # Create user
        from datetime import datetime
        user_data = {
            'username': username,
            'name': name or 'N/A',
            'phone': phone or 'N/A',
            'telegram_username': telegram_username or 'N/A',
            'key_hash': func.hash_key(access_key),
            'created_at': datetime.now().isoformat(),
            'last_used': datetime.now().isoformat(),
            'is_admin': bool(is_admin),
            'chat_id': chat_id or 'N/A',
            'ip_address': 'Unknown',
            'telegram_notification': False,
            'checks': {},
            'total_checks': 0,
            'config': {
                'use_proxy': False,
                'proxy_type': 'http',
                'proxy_host': 'proxyhost:port',
                'proxy_username': 'username',
                'proxy_password': 'password',
                'threads': 5,
                'list_file': os.path.join(ROOT_DIR, 'list.txt')
            }
        }
        
        if func.save_user_data(username, user_data):
            return jsonify({'success': True, 'message': 'User created successfully', 'username': username})
        else:
            return jsonify({'success': False, 'error': 'Failed to create user'}), 500
    except Exception as e:
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/api/admin/user/delete', methods=['POST'])
def api_admin_delete_user():
    """Delete user (admin only)"""
    try:
        data = request.json or {}
        admin_username = data.get('admin_username', 'admin')
        admin_user = func.get_user(admin_username)
        
        if not admin_user or not admin_user.get('is_admin', False):
            return jsonify({'success': False, 'error': 'Admin access required'}), 403
        
        username = data.get('username', '').strip()
        
        if not username:
            return jsonify({'success': False, 'error': 'Username is required'}), 400
        
        if username == admin_username:
            return jsonify({'success': False, 'error': 'Cannot delete your own account'}), 400
        
        if not func.user_exists(username):
            return jsonify({'success': False, 'error': 'User not found'}), 404
        
        import os
        filepath = os.path.join(func.USERS_FOLDER, f"{username}.json")
        if os.path.exists(filepath):
            os.remove(filepath)
            func.load_user_data()  # Reload cache
            return jsonify({'success': True, 'message': 'User deleted successfully'})
        else:
            return jsonify({'success': False, 'error': 'User file not found'}), 404
    except Exception as e:
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/api/admin/user/update', methods=['POST'])
def api_admin_update_user():
    """Update user (admin only)"""
    try:
        data = request.json or {}
        admin_username = data.get('admin_username', 'admin')
        admin_user = func.get_user(admin_username)
        
        if not admin_user or not admin_user.get('is_admin', False):
            return jsonify({'success': False, 'error': 'Admin access required'}), 403
        
        username = data.get('username', '').strip()
        
        if not username:
            return jsonify({'success': False, 'error': 'Username is required'}), 400
        
        user_info = func.get_user(username)
        if not user_info:
            return jsonify({'success': False, 'error': 'User not found'}), 404
        
        # Update fields
        if 'name' in data:
            user_info['name'] = data['name'].strip()
        if 'phone' in data:
            user_info['phone'] = data['phone'].strip()
        if 'telegram_username' in data:
            user_info['telegram_username'] = data['telegram_username'].strip()
        if 'chat_id' in data:
            user_info['chat_id'] = data['chat_id'].strip()
        if 'access_key' in data and data['access_key']:
            user_info['key_hash'] = func.hash_key(data['access_key'].strip())
        if 'is_admin' in data:
            if username == admin_username and not data['is_admin']:
                return jsonify({'success': False, 'error': 'Cannot remove admin status from your own account'}), 400
            user_info['is_admin'] = bool(data['is_admin'])
        if 'telegram_notification' in data:
            user_info['telegram_notification'] = bool(data['telegram_notification'])
        
        if func.save_user_data(username, user_info):
            return jsonify({'success': True, 'message': 'User updated successfully'})
        else:
            return jsonify({'success': False, 'error': 'Failed to update user'}), 500
    except Exception as e:
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/api/admin/settings', methods=['GET', 'POST'])
def api_admin_settings():
    """Get or update admin settings (admin only)"""
    try:
        admin_username = request.args.get('admin_username') or (request.json or {}).get('admin_username', 'admin')
        admin_user = func.get_user(admin_username)
        
        if not admin_user or not admin_user.get('is_admin', False):
            return jsonify({'error': 'Admin access required'}), 403
        
        if request.method == 'GET':
            # Return current settings from system_config.json and telegram_bot.py
            import json
            import os
            
            # Load from system_config.json
            config_file = os.path.join(os.path.dirname(os.path.dirname(__file__)), 'system_config.json')
            system_config = {}
            if os.path.exists(config_file):
                try:
                    with open(config_file, 'r') as f:
                        system_config = json.load(f)
                except:
                    pass
            
            # Try to get from telegram_bot.py
            try:
                from telegram_bot import BOT_TOKEN, TELEGRAM_ADMIN_CHAT_ID, TELEGRAM_BOT_USERNAME
                from telegram_bot import get_base_url
                base_url = get_base_url()
            except:
                BOT_TOKEN = system_config.get('telegram_bot_token', 'Not available')
                TELEGRAM_ADMIN_CHAT_ID = system_config.get('telegram_admin_chat_id', 'Not available')
                TELEGRAM_BOT_USERNAME = system_config.get('telegram_bot_username', 'Not available')
                base_url = system_config.get('base_url', 'http://localhost')
            
            # Use values from config file if available, otherwise from telegram_bot.py
            settings = {
                'base_url': system_config.get('base_url', base_url),
                'telegram_bot_token': system_config.get('telegram_bot_token', (BOT_TOKEN[:20] + '...' if len(str(BOT_TOKEN)) > 20 else BOT_TOKEN) if BOT_TOKEN != 'Not available' else 'Not available'),
                'telegram_admin_chat_id': system_config.get('telegram_admin_chat_id', TELEGRAM_ADMIN_CHAT_ID),
                'telegram_bot_username': system_config.get('telegram_bot_username', TELEGRAM_BOT_USERNAME)
            }
            
            return jsonify({
                'success': True,
                'settings': settings
            })
        else:
            # Update settings - save to system_config.json
            import json
            import os
            
            data = request.json or {}
            config_file = os.path.join(os.path.dirname(os.path.dirname(__file__)), 'system_config.json')
            
            # Load existing config
            system_config = {}
            if os.path.exists(config_file):
                try:
                    with open(config_file, 'r') as f:
                        system_config = json.load(f)
                except:
                    pass
            
            # Update settings
            if 'base_url' in data:
                system_config['base_url'] = data['base_url'].strip()
            if 'telegram_bot_token' in data:
                system_config['telegram_bot_token'] = data['telegram_bot_token'].strip()
            if 'telegram_admin_chat_id' in data:
                system_config['telegram_admin_chat_id'] = data['telegram_admin_chat_id'].strip()
            if 'telegram_bot_username' in data:
                system_config['telegram_bot_username'] = data['telegram_bot_username'].strip()
            
            # Keep existing python_api_url if exists
            if 'python_api_url' not in system_config:
                system_config['python_api_url'] = 'http://localhost:5001'
            
            # Add metadata
            from datetime import datetime
            system_config['updated_at'] = datetime.now().isoformat()
            system_config['updated_by'] = admin_username
            
            # Save to file
            try:
                with open(config_file, 'w') as f:
                    json.dump(system_config, f, indent=4)
                return jsonify({
                    'success': True,
                    'message': 'Settings updated successfully',
                    'settings': {
                        'base_url': system_config.get('base_url'),
                        'telegram_bot_token': system_config.get('telegram_bot_token', '')[:20] + '...' if len(system_config.get('telegram_bot_token', '')) > 20 else system_config.get('telegram_bot_token', ''),
                        'telegram_admin_chat_id': system_config.get('telegram_admin_chat_id'),
                        'telegram_bot_username': system_config.get('telegram_bot_username')
                    }
                })
            except Exception as e:
                return jsonify({'success': False, 'error': f'Failed to save settings: {str(e)}'}), 500
    except Exception as e:
        return jsonify({'error': str(e)}), 500

@app.route('/api/user/config', methods=['GET', 'POST'])
def api_user_config():
    """Get or update user config"""
    try:
        data = request.json or {}
        username = data.get('username', request.args.get('username', 'admin'))
        
        if request.method == 'GET':
            config = func.load_user_config(username)
            return jsonify(config)
        else:
            new_config = data.get('config', {})
            if not new_config:
                return jsonify({'success': False, 'error': 'Config data is required'}), 400
            
            # Ensure threads is an integer
            if 'threads' in new_config:
                try:
                    new_config['threads'] = int(new_config['threads'])
                except (ValueError, TypeError):
                    new_config['threads'] = 5
            # Ensure threads is within valid range
            if 'threads' in new_config:
                new_config['threads'] = max(1, min(50, new_config['threads']))
            
            # Ensure all config fields are properly set
            if 'use_proxy' not in new_config:
                new_config['use_proxy'] = False
            if 'proxy_type' not in new_config:
                new_config['proxy_type'] = 'http'
            if 'proxy_host' not in new_config:
                new_config['proxy_host'] = ''
            if 'proxy_username' not in new_config:
                new_config['proxy_username'] = ''
            if 'proxy_password' not in new_config:
                new_config['proxy_password'] = ''
            
            try:
                result = func.save_user_config(username, new_config)
                if result:
                    return jsonify({'success': True, 'message': 'Config updated'})
                else:
                    return jsonify({'success': False, 'error': 'Failed to save config. User not found or save failed.'}), 500
            except Exception as e:
                return jsonify({'success': False, 'error': f'Error saving config: {str(e)}'}), 500
    except Exception as e:
        return jsonify({'error': str(e)}), 500

@app.route('/api/user/dashboard', methods=['GET'])
def api_user_dashboard():
    """Get dashboard statistics (optionally filtered by service)"""
    try:
        username = request.args.get('username', 'admin')
        service = request.args.get('service', None)  # Optional service filter
        
        user_info = func.get_user(username) or {}
        checks = user_info.get('checks', {})
        
        # Filter by service if provided
        if service:
            checks = {k: v for k, v in checks.items() if v.get('service') == service}
        
        total_checks = len(checks)
        active_checks = sum(1 for c in checks.values() if c.get('status') == 'running')
        total_success = sum(c.get('success', 0) for c in checks.values())
        total_failed = sum(c.get('failed', 0) for c in checks.values())
        total_accounts = total_success + total_failed
        success_rate = (total_success / total_accounts * 100) if total_accounts > 0 else 0
        
        return jsonify({
            'total_checks': total_checks,
            'active_checks': active_checks,
            'total_success': total_success,
            'total_failed': total_failed,
            'success_rate': round(success_rate, 2),
            'service': service or 'All Services'
        })
    except Exception as e:
        return jsonify({'error': str(e)}), 500

@app.route('/api/user/history', methods=['GET'])
def api_user_history():
    """Get user check history"""
    try:
        username = request.args.get('username', 'admin')
        user_info = func.get_user(username) or {}
        checks = user_info.get('checks', {})
        
        # Convert checks dict to list sorted by start_time (newest first)
        history = []
        for check_id, check_data in checks.items():
            history.append({
                'check_id': check_id,
                'status': check_data.get('status', 'unknown'),
                'progress': check_data.get('progress', 0),
                'total_accounts': check_data.get('total_accounts', 0),
                'completed_accounts': check_data.get('completed_accounts', 0),
                'success': check_data.get('success', 0),
                'failed': check_data.get('failed', 0),
                'start_time': check_data.get('start_time'),
                'end_time': check_data.get('end_time'),
                'checking_ip': check_data.get('checking_ip', 'Unknown'),
                'service': check_data.get('service', 'Microsoft Account Checker')
            })
        
        # Sort by start_time descending (newest first)
        history.sort(key=lambda x: x.get('start_time') or '', reverse=True)
        
        return jsonify({'history': history})
    except Exception as e:
        return jsonify({'error': str(e)}), 500

@app.route('/api/user/history/delete', methods=['POST'])
def api_delete_history():
    """Delete a specific check from history"""
    try:
        data = request.json or {}
        username = data.get('username', request.args.get('username', 'admin'))
        check_id = data.get('check_id', '')
        
        if not check_id:
            return jsonify({'success': False, 'error': 'Check ID is required'}), 400
        
        user_info = func.get_user(username)
        if not user_info:
            return jsonify({'success': False, 'error': 'User not found'}), 404
        
        if 'checks' in user_info and check_id in user_info['checks']:
            # Delete folder and files
            check_folder = os.path.join(RESULTS_FOLDER, check_id)
            if os.path.exists(check_folder):
                try:
                    shutil.rmtree(check_folder)
                except Exception as e:
                    print(f"Error deleting folder {check_folder}: {e}")
            
            # Delete from user data
            del user_info['checks'][check_id]
            func.save_user_data(username, user_info)
            
            # Also remove from memory if exists
            if check_id in check_status:
                del check_status[check_id]
            if check_id in check_results:
                del check_results[check_id]
            
            return jsonify({'success': True, 'message': 'Check deleted successfully'})
        else:
            return jsonify({'success': False, 'error': 'Check not found'}), 404
    except Exception as e:
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/api/user/history/delete-all', methods=['POST'])
def api_delete_all_history():
    """Delete all checks from history"""
    try:
        data = request.json or {}
        username = data.get('username', request.args.get('username', 'admin'))
        
        user_info = func.get_user(username)
        if not user_info:
            return jsonify({'success': False, 'error': 'User not found'}), 404
        
        # Get all check IDs before deleting
        check_ids = list(user_info.get('checks', {}).keys())
        
        # Delete all folders and files
        deleted_folders = 0
        for check_id in check_ids:
            check_folder = os.path.join(RESULTS_FOLDER, check_id)
            if os.path.exists(check_folder):
                try:
                    shutil.rmtree(check_folder)
                    deleted_folders += 1
                except Exception as e:
                    print(f"Error deleting folder {check_folder}: {e}")
        
        # Clear all checks from user data
        user_info['checks'] = {}
        func.save_user_data(username, user_info)
        
        # Remove from memory
        for check_id in check_ids:
            if check_id in check_status:
                del check_status[check_id]
            if check_id in check_results:
                del check_results[check_id]
        
        return jsonify({
            'success': True, 
            'message': f'All {len(check_ids)} checks deleted successfully',
            'deleted_folders': deleted_folders
        })
    except Exception as e:
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/api/captcha/generate', methods=['GET'])
def api_captcha_generate():
    """Generate CAPTCHA"""
    try:
        captcha = func.generate_captcha()
        return jsonify(captcha)
    except Exception as e:
        return jsonify({'error': str(e)}), 500

@app.route('/api/otp/resend', methods=['POST'])
def api_otp_resend():
    """Resend OTP"""
    try:
        data = request.json
        username = data.get('username', '').strip()
        
        user_info = func.get_user(username)
        if not user_info:
            return jsonify({'success': False, 'error': 'User not found'}), 404
        
        chat_id = user_info.get('chat_id')
        if not chat_id:
            return jsonify({'success': False, 'error': 'Chat ID not found'}), 404
        
        func.send_otp(username, chat_id)
        return jsonify({'success': True, 'message': 'OTP sent'})
    except Exception as e:
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/api/files/<check_id>/<file_type>', methods=['GET'])
def api_files(check_id, file_type):
    """Get result files (live, dead, error)"""
    try:
        if file_type not in ['live', 'dead', 'error']:
            return jsonify({'error': 'Invalid file type'}), 400
        
        file_path = os.path.join(RESULTS_FOLDER, check_id, f'{file_type}.txt')
        if not os.path.exists(file_path):
            return jsonify({'content': ''})
        
        with open(file_path, 'r', encoding='utf-8') as f:
            content = f.read()
        
        return jsonify({'content': content})
    except Exception as e:
        return jsonify({'error': str(e)}), 500

# ═══════════════════════════════════════════════════════════════════
#                          CHECKER FUNCTION
# ═══════════════════════════════════════════════════════════════════

def run_checker(check_id, accounts, config, username):
    """Run account checker"""
    service = config.get('service', 'Microsoft Account Checker')
    use_proxy = config.get('use_proxy', False)
    proxy = None
    if use_proxy:
        proxy = api_microsoft.build_proxy_url(config)
    
    # Get threads from config, ensure it's an integer
    thread_count = config.get('threads', 1)
    if isinstance(thread_count, str):
        try:
            thread_count = int(thread_count)
        except (ValueError, TypeError):
            thread_count = 1
    elif not isinstance(thread_count, int):
        thread_count = 1
    
    # Ensure thread_count is at least 1 and at most 50
    thread_count = max(1, min(50, int(thread_count)))
    
    # Ensure start_time is set
    from datetime import datetime
    if 'start_time' not in check_status[check_id] or check_status[check_id]['start_time'] is None:
        check_status[check_id]['start_time'] = datetime.now().isoformat()
    check_results[check_id]['start_time'] = check_status[check_id]['start_time']
    
    # Determine which checker to use based on service
    is_cpanel = 'cpanel' in service.lower() or 'cPanel' in service
    
    def callback_progress(account_info, ip):
        """Progress callback - account_info can be email (Microsoft) or username@host (cPanel)"""
        if check_id in check_status:
            check_status[check_id]['checking_ip'] = ip
    
    def callback_success_microsoft(email, password, points, name, birthday, region, ip):
        """Success callback for Microsoft"""
        result = {
            'email': email,
            'password': password,
            'status': 'success',
            'points': points,
            'name': name,
            'birthday': birthday,
            'region': region,
            'ip': ip,
            'timestamp': None
        }
        
        if check_id in check_results:
            check_results[check_id]['results'].append(result)
        
        if check_id in check_status:
            check_status[check_id]['success'] = check_status[check_id].get('success', 0) + 1
            check_status[check_id]['completed_accounts'] = check_status[check_id].get('completed_accounts', 0) + 1
            total = check_status[check_id].get('total_accounts', 1)
            check_status[check_id]['progress'] = int((check_status[check_id]['completed_accounts'] / total) * 100)
            
            # Save progress periodically
            if check_status[check_id]['completed_accounts'] % 5 == 0:
                func.save_check_progress(
                    username,
                    check_id,
                    check_status[check_id],
                    check_results.get(check_id, {})
                )
    
    def callback_success_cpanel(host, username_cp, password, account_type, reseller_count, email_success, 
                                domains, emails, successful_domains, failed_domains, ip):
        """Success callback for cPanel"""
        result = {
            'host': host,
            'username': username_cp,
            'password': password,
            'status': 'success',
            'account_type': account_type,
            'reseller_count': reseller_count,
            'email_success': email_success,
            'domains': domains,
            'emails': emails,
            'successful_domains': successful_domains,
            'failed_domains': failed_domains,
            'ip': ip,
            'timestamp': None
        }
        
        if check_id in check_results:
            check_results[check_id]['results'].append(result)
        
        if check_id in check_status:
            check_status[check_id]['success'] = check_status[check_id].get('success', 0) + 1
            check_status[check_id]['completed_accounts'] = check_status[check_id].get('completed_accounts', 0) + 1
            total = check_status[check_id].get('total_accounts', 1)
            check_status[check_id]['progress'] = int((check_status[check_id]['completed_accounts'] / total) * 100)
            
            # Save progress periodically
            if check_status[check_id]['completed_accounts'] % 5 == 0:
                func.save_check_progress(
                    username,
                    check_id,
                    check_status[check_id],
                    check_results.get(check_id, {})
                )
    
    def callback_error_microsoft(email, password, error_message, ip, is_dead):
        """Error callback for Microsoft"""
        result = {
            'email': email,
            'password': password,
            'status': 'failed',
            'message': error_message,
            'ip': ip,
            'timestamp': None
        }
        
        if check_id in check_results:
            check_results[check_id]['results'].append(result)
        
        if check_id in check_status:
            check_status[check_id]['failed'] = check_status[check_id].get('failed', 0) + 1
            check_status[check_id]['completed_accounts'] = check_status[check_id].get('completed_accounts', 0) + 1
            total = check_status[check_id].get('total_accounts', 1)
            check_status[check_id]['progress'] = int((check_status[check_id]['completed_accounts'] / total) * 100)
            
            # Save progress periodically
            if check_status[check_id]['completed_accounts'] % 5 == 0:
                func.save_check_progress(
                    username,
                    check_id,
                    check_status[check_id],
                    check_results.get(check_id, {})
                )
    
    def callback_error_cpanel(host, username_cp, password, error_message, ip, is_dead):
        """Error callback for cPanel"""
        result = {
            'host': host,
            'username': username_cp,
            'password': password,
            'status': 'failed',
            'message': error_message,
            'ip': ip,
            'timestamp': None
        }
        
        if check_id in check_results:
            check_results[check_id]['results'].append(result)
        
        if check_id in check_status:
            check_status[check_id]['failed'] = check_status[check_id].get('failed', 0) + 1
            check_status[check_id]['completed_accounts'] = check_status[check_id].get('completed_accounts', 0) + 1
            total = check_status[check_id].get('total_accounts', 1)
            check_status[check_id]['progress'] = int((check_status[check_id]['completed_accounts'] / total) * 100)
            
            # Save progress periodically
            if check_status[check_id]['completed_accounts'] % 5 == 0:
                func.save_check_progress(
                    username,
                    check_id,
                    check_status[check_id],
                    check_results.get(check_id, {})
                )
    
    # Run checker with ThreadPoolExecutor
    with ThreadPoolExecutor(max_workers=thread_count) as executor:
        futures = []
        for index, account in enumerate(accounts, start=1):
            # Check if stopped
            if check_id in check_status and check_status[check_id]['status'] != 'running':
                break
            
            if is_cpanel:
                # cPanel format: [host, username, password]
                if len(account) >= 3:
                    host, username_cp, password = account[0], account[1], account[2]
                else:
                    continue
                
                test_email = config.get('test_email', 'test@example.com')
                future = executor.submit(
                    api_cpanel.check_account,
                    host, username_cp, password, use_proxy, proxy,
                    callback_success=callback_success_cpanel,
                    callback_error=callback_error_cpanel,
                    callback_progress=callback_progress,
                    check_id=check_id,
                    results_folder=RESULTS_FOLDER,
                    test_email=test_email
                )
            else:
                # Microsoft format: [email, password]
                email, password = account
                future = executor.submit(
                    api_microsoft.check_account,
                    email, password, use_proxy, proxy,
                    callback_success=callback_success_microsoft,
                    callback_error=callback_error_microsoft,
                    callback_progress=callback_progress,
                    check_id=check_id,
                    results_folder=RESULTS_FOLDER
                )
            futures.append(future)
        
        # Wait for completion
        for future in as_completed(futures):
            try:
                result = future.result()  # This will raise exception if there was an error
                logger.info(f"[CHECKER] Account check completed: {result.get('status', 'unknown')}")
            except Exception as e:
                logger.error(f"[ERROR] Exception in checker thread: {e}")
                import traceback
                traceback.print_exc()
                # Continue to next future
                continue
            
            if check_id in check_status and check_status[check_id]['status'] != 'running':
                # Cancel remaining futures
                for f in futures:
                    if not f.done():
                        f.cancel()
                break
    
    # Final status update
    if check_id in check_status:
        from datetime import datetime
        if check_status[check_id]['status'] == 'running':
            check_status[check_id]['status'] = 'completed'
        elif check_status[check_id]['status'] == 'stopped':
            check_status[check_id]['status'] = 'stopped'
        
        # Ensure start_time is set
        if 'start_time' not in check_status[check_id] or check_status[check_id]['start_time'] is None:
            check_status[check_id]['start_time'] = datetime.now().isoformat()
        
        # Set end_time
        check_status[check_id]['end_time'] = datetime.now().isoformat()
        
        # Save final progress
        func.save_check_progress(
            username,
            check_id,
            check_status[check_id],
            check_results.get(check_id, {})
        )
        
        # Update user data
        user_info = func.get_user(username)
        if user_info:
            if 'checks' not in user_info:
                user_info['checks'] = {}
            user_info['checks'][check_id] = check_status[check_id].copy()
            func.save_user_data(username, user_info)
        
        # Send notification if enabled
        # Get base URL for notification
        base_url = "http://localhost"
        try:
            from telegram_bot import get_base_url
            base_url = get_base_url()
        except:
            pass
        
        status = check_status[check_id].get('status', 'completed')
        service_name = check_status[check_id].get('service', 'Microsoft Account Checker')
        total = check_status[check_id].get('total_accounts', 0)
        success = check_status[check_id].get('success', 0)
        failed = check_status[check_id].get('failed', 0)
        progress = check_status[check_id].get('progress', 0)
        completed = check_status[check_id].get('completed_accounts', 0)
        
        if status == 'completed':
            notification_msg = f"""✅ <b>Check Completed</b>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

🎉 <b>Your check has been completed!</b>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

📋 <b>Check Results:</b>

🔹 <b>Service:</b> {service_name}
🔹 <b>Total Accounts:</b> {total}
🔹 <b>Success:</b> {success}
🔹 <b>Failed:</b> {failed}
🔹 <b>Progress:</b> {progress}%

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

🆔 <b>Check ID:</b>
<code>{check_id}</code>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

📊 <b>Success Rate:</b> {round((success / total * 100) if total > 0 else 0, 2)}%

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

🌐 <b>View Results:</b>
{base_url}/php/monitor.php?check_id={check_id}

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"""
        elif status == 'stopped':
            notification_msg = f"""⏹️ <b>Check Stopped</b>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

⚠️ <b>Your check has been stopped!</b>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

📋 <b>Check Status:</b>

🔹 <b>Service:</b> {service_name}
🔹 <b>Total Accounts:</b> {total}
🔹 <b>Completed:</b> {completed}
🔹 <b>Success:</b> {success}
🔹 <b>Failed:</b> {failed}
🔹 <b>Progress:</b> {progress}%

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

🆔 <b>Check ID:</b>
<code>{check_id}</code>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

🌐 <b>View Results:</b>
{base_url}/php/monitor.php?check_id={check_id}

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"""
        else:
            notification_msg = None
        
        if notification_msg:
            func.send_user_notification(username, notification_msg)

# ═══════════════════════════════════════════════════════════════════
#                          MAIN FUNCTION
# ═══════════════════════════════════════════════════════════════════

if __name__ == '__main__':
    # Load user data on startup
    func.load_user_data()
    
    # Get port from environment or default to 5001
    port = int(os.getenv('API_PORT', '5001'))
    
    print(f"\n[API] Starting Python API Server on port {port}...")
    print(f"[API] API will be available at: http://localhost:{port}\n")
    
    app.run(host='0.0.0.0', port=port, debug=False)

# For cPanel/Passenger deployment
# The 'application' variable is required for Passenger WSGI
application = app

