"""
User Management Functions
Handles login, registration, and user management with Telegram Bot integration
"""
import os
import json
import hashlib
import uuid
import requests
from datetime import datetime
from flask import session, request

# Import Telegram Bot functions from telegram_bot.py
try:
    import telegram_bot
    # Use Telegram configuration from telegram_bot module
    TELEGRAM_BOT_TOKEN = telegram_bot.BOT_TOKEN
    TELEGRAM_API_URL = telegram_bot.TELEGRAM_API_URL
    TELEGRAM_ADMIN_CHAT_ID = telegram_bot.TELEGRAM_ADMIN_CHAT_ID
    TELEGRAM_BOT_USERNAME = telegram_bot.TELEGRAM_BOT_USERNAME
except ImportError:
    # Fallback if telegram_bot.py not available
    TELEGRAM_BOT_TOKEN = "8283589446:AAFcoaT-VMrQiFutqRVA2Ty-Sr0P5bxB990"
    TELEGRAM_API_URL = f"https://api.telegram.org/bot{TELEGRAM_BOT_TOKEN}"
    TELEGRAM_ADMIN_CHAT_ID = "1859095647"
    TELEGRAM_BOT_USERNAME = "forbidden_tools_bot"

# User data storage - relative to public_html directory
SCRIPT_DIR = os.path.dirname(os.path.abspath(__file__))
PARENT_DIR = os.path.dirname(SCRIPT_DIR)  # php/ directory

# Find public_html or public_htmls directory (for cPanel compatibility)
ROOT_DIR = None
parts = SCRIPT_DIR.split(os.sep)

# Check for public_html or public_htmls in path
for folder_name in ['public_htmls', 'public_html']:
    if folder_name in parts:
        idx = parts.index(folder_name)
        ROOT_DIR = os.sep.join(parts[:idx+1])
        break

# If not found, try to find by going up from current directory
if ROOT_DIR is None:
    current = SCRIPT_DIR
    max_levels = 5  # Don't go too far up
    for _ in range(max_levels):
        parent = os.path.dirname(current)
        if os.path.basename(parent) in ['public_htmls', 'public_html']:
            ROOT_DIR = parent
            break
        if parent == current:  # Reached root
            break
        current = parent

# If still not found, use parent directory (php/ -> web root)
if ROOT_DIR is None:
    ROOT_DIR = os.path.dirname(PARENT_DIR)

USERS_FOLDER = os.path.join(ROOT_DIR, "users")
# Only create folder if we have write permission
try:
    os.makedirs(USERS_FOLDER, exist_ok=True)
except (PermissionError, OSError) as e:
    # If can't create in ROOT_DIR, try creating in SCRIPT_DIR
    USERS_FOLDER = os.path.join(SCRIPT_DIR, "users")
    try:
        os.makedirs(USERS_FOLDER, exist_ok=True)
    except (PermissionError, OSError):
        # Last resort: use current directory
        USERS_FOLDER = os.path.join(os.getcwd(), "users")
        os.makedirs(USERS_FOLDER, exist_ok=True)

# Base URL helper
def get_base_url():
    """Get base URL (http or https) based on configuration"""
    # Note: This is mainly for Telegram bot messages, PHP web interface doesn't use this
    use_https = os.getenv('USE_HTTPS', 'false').lower() == 'true'
    if use_https:
        port = int(os.getenv('HTTPS_PORT', '443'))
        if port == 443:
            return "https://localhost"
        return f"https://localhost:{port}"
    else:
        port = int(os.getenv('HTTP_PORT', '5000'))
        if port == 80:
            return "http://localhost"
        return f"http://localhost:{port}"

# Global user data cache
user_data_cache = {}

# OTP storage (in-memory, should be moved to database in production)
otp_storage = {}  # {username: {'otp': str, 'expires': datetime, 'attempts': int}}


def hash_key(key):
    """Hash access key using SHA256"""
    return hashlib.sha256(key.encode()).hexdigest()


def load_user_data():
    """Load all user data from JSON files"""
    global user_data_cache
    user_data_cache = {}
    
    if not os.path.exists(USERS_FOLDER):
        os.makedirs(USERS_FOLDER, exist_ok=True)
        return user_data_cache
    
    for filename in os.listdir(USERS_FOLDER):
        if filename.endswith('.json'):
            username = filename[:-5]  # Remove .json extension
            filepath = os.path.join(USERS_FOLDER, filename)
            try:
                with open(filepath, 'r', encoding='utf-8') as f:
                    user_data_cache[username] = json.load(f)
            except Exception as e:
                print(f"Error loading user {username}: {e}")
    
    return user_data_cache


def save_user_data(username, user_data):
    """Save user data to JSON file"""
    filepath = os.path.join(USERS_FOLDER, f"{username}.json")
    try:
        with open(filepath, 'w', encoding='utf-8') as f:
            json.dump(user_data, f, indent=2, ensure_ascii=False)
        user_data_cache[username] = user_data
        return True
    except Exception as e:
        print(f"Error saving user {username}: {e}")
        return False


def get_user(username):
    """Get user data by username"""
    if not username:
        print("Error: Username is empty")
        return None
    
    # Check cache first
    if username in user_data_cache:
        return user_data_cache[username]
    
    # Try to load from file
    filepath = os.path.join(USERS_FOLDER, f"{username}.json")
    if os.path.exists(filepath):
        try:
            with open(filepath, 'r', encoding='utf-8') as f:
                user_data = json.load(f)
                user_data_cache[username] = user_data
                return user_data
        except Exception as e:
            print(f"Error loading user {username} from file: {e}")
            import traceback
            traceback.print_exc()
    else:
        print(f"User file not found: {filepath}")
        # Try to reload all users in case cache is stale
        load_user_data()
        if username in user_data_cache:
            return user_data_cache[username]
    
    return None


def user_exists(username):
    """Check if user exists"""
    return username in user_data_cache or os.path.exists(
        os.path.join(USERS_FOLDER, f"{username}.json")
    )


def verify_telegram_user(telegram_id, username=None):
    """Verify Telegram user with bot - uses telegram_bot.py"""
    try:
        return telegram_bot.verify_telegram_user(telegram_id, username)
    except (ImportError, AttributeError):
        # Fallback if telegram_bot not available
        try:
            url = f"{TELEGRAM_API_URL}/getChat"
            params = {"chat_id": telegram_id}
            response = requests.get(url, params=params, timeout=10)
            
            if response.status_code == 200:
                data = response.json()
                if data.get('ok'):
                    chat = data.get('result', {})
                    if username:
                        if chat.get('username', '').lower() == username.lower():
                            return True
                    if chat.get('first_name') or chat.get('last_name'):
                        return True
            return False
        except Exception as e:
            print(f"Error verifying Telegram user: {e}")
            return False


def send_telegram_message(chat_id, message):
    """Send message via Telegram Bot API - uses telegram_bot.py"""
    try:
        return telegram_bot.send_telegram_message(chat_id, message)
    except (ImportError, AttributeError):
        # Fallback if telegram_bot not available
        try:
            url = f"{TELEGRAM_API_URL}/sendMessage"
            params = {
                "chat_id": chat_id,
                "text": message,
                "parse_mode": "HTML"
            }
            response = requests.post(url, params=params, timeout=10)
            result = response.json()
            
            if response.status_code == 200 and result.get('ok'):
                print(f"[TELEGRAM] Successfully sent message to chat_id: {chat_id}")
                return True
            else:
                print(f"[TELEGRAM] Failed to send message to chat_id: {chat_id}, response: {result}")
                return False
        except Exception as e:
            print(f"[TELEGRAM] Error sending Telegram message to chat_id {chat_id}: {e}")
            import traceback
            traceback.print_exc()
            return False


def send_user_notification(username, message):
    """
    Send notification to user via Telegram if notification is enabled
    
    Args:
        username: Username
        message: Message to send
    
    Returns:
        bool: True if sent successfully, False otherwise
    """
    try:
        user_info = get_user(username)
        if not user_info:
            return False
        
        # Check if notification is enabled
        if not user_info.get('telegram_notification', False):
            return False
        
        # Get chat_id
        chat_id = user_info.get('chat_id') or user_info.get('telegram_id')
        if not chat_id:
            return False
        
        # Send notification
        return send_telegram_message(chat_id, message)
    except Exception as e:
        print(f"[NOTIFICATION] Error sending notification to {username}: {e}")
        return False


def register_user_via_telegram(chat_id, captcha_code=None, captcha_answer=None):
    """
    Register a new user via Telegram Bot (credentials generated by bot) using Chat ID
    NOTE: This is now handled directly by Telegram bot conversation handler
    
    Args:
        chat_id: Telegram Chat ID (required)
        captcha_code: CAPTCHA code from session (deprecated)
        captcha_answer: User's answer to CAPTCHA (deprecated)
    
    Returns:
        dict: {'success': bool, 'username': str, 'access_key': str, 'message': str}
    """
    # Validate Chat ID
    if not chat_id:
        return {'success': False, 'message': f'Chat ID is required. Please contact admin (Chat ID: {TELEGRAM_ADMIN_CHAT_ID})'}
    
    # Check if user already registered
    load_user_data()
    for username, user_info in user_data_cache.items():
        if user_info.get('chat_id') == str(chat_id):
            return {
                'success': False,
                'message': 'You are already registered! Please use the credentials that were sent previously or register via Telegram bot with /start'
            }
    
    # Return message to use Telegram bot
    return {
        'success': False,
        'message': 'Registration must be done via Telegram bot. Please send /start to @forbidden_tools_bot to start registration.'
    }


def register_user(username, access_key, telegram_id=None, telegram_username=None, captcha_code=None, captcha_answer=None):
    """
    Register a new user with Telegram verification and CAPTCHA
    
    Args:
        username: Username for the account
        access_key: Access key/password
        telegram_id: Telegram user ID (optional)
        telegram_username: Telegram username (optional)
        captcha_code: CAPTCHA code from session
        captcha_answer: User's answer to CAPTCHA
    
    Returns:
        dict: {'success': bool, 'message': str}
    """
    # Validate inputs
    if not username or not access_key:
        return {'success': False, 'message': 'Username and access key are required'}
    
    if len(username) < 3:
        return {'success': False, 'message': 'Username must be at least 3 characters'}
    
    if len(access_key) < 6:
        return {'success': False, 'message': 'Access key must be at least 6 characters'}
    
    # Check if user already exists
    if user_exists(username):
        return {'success': False, 'message': 'Username already exists'}
    
    # Verify CAPTCHA
    if not verify_captcha(captcha_code, captcha_answer):
        return {'success': False, 'message': 'Invalid CAPTCHA code'}
    
    # Verify Telegram if provided
    if telegram_id:
        if not verify_telegram_user(telegram_id, telegram_username or username):
            return {'success': False, 'message': 'Telegram verification failed'}
    
    # Create user data
    user_data = {
        'username': username,
        'key_hash': hash_key(access_key),
        'created_at': datetime.now().isoformat(),
        'last_used': datetime.now().isoformat(),
        'is_admin': False,
        'telegram_id': telegram_id,
        'telegram_username': telegram_username,
        'telegram_notification': False,  # Default: notification disabled
        'ip_address': 'Unknown',  # IP Address
        'checks': {},
        'total_checks': 0,
        'config': {
            'use_proxy': False,
            'proxy_type': 'http',
            'proxy_host': 'proxyhost:port',
            'proxy_username': 'username',
            'proxy_password': 'password',
            'threads': 5,
            'list_file': os.path.join(ROOT_DIR, 'list.txt')
        }
    }
    
    # Save user data
    if save_user_data(username, user_data):
        # Send welcome message via Telegram if available
        if telegram_id:
            welcome_msg = f"""🎉 <b>Registration Successful!</b>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

👋 <b>Welcome to 403 Forbidden, {username}!</b>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

✅ <b>Your account has been created!</b>
You can now login using your access key.

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

👤 <b>Account Credentials:</b>

🔹 <b>Username:</b>
<code>{username}</code>

🔑 <b>Access Key:</b>
<code>{access_key}</code>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

⚠️ <b>Security Reminder:</b>
Save your access key securely!

🌐 <b>Login:</b>
{get_base_url()}/login

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"""
            send_telegram_message(telegram_id, welcome_msg)
        
        return {'success': True, 'message': 'Registration successful'}
    else:
        return {'success': False, 'message': 'Failed to save user data'}


def login_user_step1(username_or_key):
    """
    Step 1: Verify username or access key and send OTP
    
    Args:
        username_or_key: Username or access key/password
    
    Returns:
        dict: {'success': bool, 'username': str, 'message': str}
    """
    if not username_or_key:
        return {'success': False, 'message': 'Username or access key is required'}
    
    username = None
    access_key = None
    
    load_user_data()
    
    # Try to find by username first
    if username_or_key in user_data_cache:
        username = username_or_key
        user_info = user_data_cache[username]
        # Username found, we'll need access_key in step 2
    else:
        # Try to find by access key hash
        key_hash = hash_key(username_or_key)
        for user, user_info in user_data_cache.items():
            if user_info.get('key_hash') == key_hash:
                username = user
                access_key = username_or_key
                break
    
    if not username:
        return {'success': False, 'message': 'Invalid username or access key'}
    
    # Send OTP
    otp_result = send_otp(username)
    if otp_result['success']:
        return {
            'success': True,
            'username': username,
            'message': 'OTP sent to your Telegram. Please check your messages.'
        }
    else:
        return {
            'success': False,
            'message': otp_result['message']
        }


def login_user_step2(username, access_key_or_username, otp_code):
    """
    Step 2: Verify OTP and complete login
    
    Args:
        username: Username (from step 1)
        access_key_or_username: Access key/password OR username (if user logged in with username in step 1)
        otp_code: OTP code
    
    Returns:
        dict: {'success': bool, 'message': str, 'user_data': dict}
    """
    if not username or not otp_code:
        return {'success': False, 'message': 'Username and OTP are required'}
    
    user_data = get_user(username)
    if not user_data:
        return {'success': False, 'message': 'Invalid username'}
    
    # If access_key_or_username is provided, verify it
    # If it's the same as username, skip access_key verification (user logged in with username)
    # If it's different, verify it as access_key
    if access_key_or_username:
        if access_key_or_username != username:
            # Verify access key
            key_hash = hash_key(access_key_or_username)
            if user_data.get('key_hash') != key_hash:
                return {'success': False, 'message': 'Invalid access key'}
        # If access_key_or_username == username, user logged in with username, skip access_key verification
    
    # Verify OTP
    otp_result = verify_otp(username, otp_code)
    if not otp_result['success']:
        return {
            'success': False,
            'message': otp_result['message']
        }
    
    # Update last used timestamp and IP address
    user_data['last_used'] = datetime.now().isoformat()
    
    # Get IP address from request if available
    try:
        if request:
            # Try to get real IP (considering proxies)
            ip_address = request.headers.get('X-Forwarded-For', request.headers.get('X-Real-IP', request.remote_addr))
            if ip_address:
                # Take first IP if comma-separated (from X-Forwarded-For)
                ip_address = ip_address.split(',')[0].strip()
                user_data['ip_address'] = ip_address
    except Exception as e:
        print(f"[FUNC] Error getting IP address: {e}")
        # Keep existing IP or set to Unknown
        if 'ip_address' not in user_data:
            user_data['ip_address'] = 'Unknown'
    
    save_user_data(username, user_data)
    
    return {
        'success': True,
        'message': 'Login successful',
        'user_data': user_data
    }


def login_user(username, access_key):
    """
    Legacy login function (for backward compatibility)
    Now requires OTP - use login_user_step1 and login_user_step2 instead
    """
    # This is now a 2-step process, but we keep this for compatibility
    # Step 1: Send OTP
    step1 = login_user_step1(access_key)
    if not step1['success']:
        return step1
    
    # For legacy support, we can't complete login without OTP
    return {
        'success': False,
        'message': 'OTP verification required. Please use the new login flow.',
        'requires_otp': True,
        'username': step1.get('username')
    }


def verify_captcha(captcha_code, captcha_answer):
    """
    Verify CAPTCHA answer
    
    Args:
        captcha_code: CAPTCHA code stored in session
        captcha_answer: User's answer
    
    Returns:
        bool: True if CAPTCHA is correct
    """
    if not captcha_code or not captcha_answer:
        return False
    
    # Simple CAPTCHA verification (case-insensitive, strip whitespace)
    return captcha_code.lower().strip() == captcha_answer.lower().strip()


def generate_captcha():
    """
    Generate a simple CAPTCHA code
    
    Returns:
        dict: {'code': str, 'question': str}
    """
    import random
    
    # Simple math CAPTCHA
    num1 = random.randint(1, 10)
    num2 = random.randint(1, 10)
    operation = random.choice(['+', '-', '*'])
    
    if operation == '+':
        answer = num1 + num2
        question = f"{num1} + {num2} = ?"
    elif operation == '-':
        # Ensure positive result
        if num1 < num2:
            num1, num2 = num2, num1
        answer = num1 - num2
        question = f"{num1} - {num2} = ?"
    else:  # *
        answer = num1 * num2
        question = f"{num1} × {num2} = ?"
    
    return {
        'code': str(answer),
        'question': question
    }


def update_user_data(username, updates):
    """Update user data with provided updates"""
    user_data = get_user(username)
    if not user_data:
        return False
    
    user_data.update(updates)
    return save_user_data(username, user_data)


def delete_user(username):
    """Delete user account"""
    if username in user_data_cache:
        del user_data_cache[username]
    
    filepath = os.path.join(USERS_FOLDER, f"{username}.json")
    if os.path.exists(filepath):
        try:
            os.remove(filepath)
            return True
        except Exception as e:
            print(f"Error deleting user {username}: {e}")
            return False
    
    return False


def get_all_users():
    """Get list of all users"""
    load_user_data()
    return list(user_data_cache.keys())


def is_admin(username):
    """Check if user is admin"""
    user_data = get_user(username)
    if user_data:
        return user_data.get('is_admin', False)
    return False


def create_default_admin(username="admin", access_key="admin-key"):
    """Create default admin user with admin Chat ID"""
    if user_exists(username):
        return False
    
    admin_data = {
        'username': username,
        'key_hash': hash_key(access_key),
        'created_at': datetime.now().isoformat(),
        'last_used': datetime.now().isoformat(),
        'is_admin': True,
        'chat_id': TELEGRAM_ADMIN_CHAT_ID,
        'telegram_notification': False,  # Default: notification disabled
        'checks': {},
        'total_checks': 0,
        'config': {
            'use_proxy': False,
            'proxy_type': 'http',
            'proxy_host': 'proxyhost:port',
            'proxy_username': 'username',
            'proxy_password': 'password',
            'threads': 5,
            'list_file': os.path.join(ROOT_DIR, 'list.txt')
        }
    }
    
    return save_user_data(username, admin_data)


def load_user_config(username):
    """Load user-specific config from user data"""
    user_info = get_user(username)
    if user_info and 'config' in user_info:
        return user_info['config']
    
    # Return default config if no user config exists
    default_config = {
        'use_proxy': False,
        'proxy_type': 'http',
        'proxy_host': 'proxyhost:port',
        'proxy_username': 'username',
        'proxy_password': 'password',
        'threads': 5,
        'list_file': 'list.txt'
    }
    return default_config


def save_user_config(username, config):
    """Save user-specific config to user data"""
    try:
        if not username:
            print("Error: Username is empty in save_user_config")
            return False
        
        print(f"Attempting to save config for user: {username}")
        
        # Try to reload user data in case cache is stale
        load_user_data()
        
        user_info = get_user(username)
        if not user_info:
            print(f"Error: User '{username}' not found. Available users: {list(user_data_cache.keys())}")
            # Try to create default config if user file exists but wasn't loaded
            filepath = os.path.join(USERS_FOLDER, f"{username}.json")
            if os.path.exists(filepath):
                print(f"User file exists but couldn't be loaded: {filepath}")
            return False
        
        # Ensure threads is an integer
        if 'threads' in config:
            try:
                config['threads'] = int(config['threads'])
            except (ValueError, TypeError):
                config['threads'] = 5  # Default to 5 if invalid
        else:
            config['threads'] = 5  # Default to 5 if not provided
        
        # Ensure threads is within valid range
        config['threads'] = max(1, min(50, config['threads']))
        
        # Ensure all required config fields exist
        if 'use_proxy' not in config:
            config['use_proxy'] = False
        if 'proxy_type' not in config:
            config['proxy_type'] = 'http'
        if 'proxy_host' not in config:
            config['proxy_host'] = ''
        if 'proxy_username' not in config:
            config['proxy_username'] = ''
        if 'proxy_password' not in config:
            config['proxy_password'] = ''
        
        user_info['config'] = config
        result = save_user_data(username, user_info)
        if not result:
            print(f"Error: Failed to save user data for '{username}'")
        else:
            print(f"Successfully saved config for user: {username}")
        return result
    except Exception as e:
        print(f"Error in save_user_config for '{username}': {e}")
        import traceback
        traceback.print_exc()
        return False


def save_check_progress(username, check_id, check_status_data, check_results_data):
    """
    Save check progress to user data
    
    Args:
        username: Username
        check_id: Check ID
        check_status_data: Status data from check_status
        check_results_data: Results data from check_results
    """
    user_info = get_user(username)
    if not user_info:
        return False
    
    if 'checks' not in user_info:
        user_info['checks'] = {}
    
    if check_id not in user_info['checks']:
        user_info['checks'][check_id] = {}
    
    # Update check data
    user_info['checks'][check_id].update({
        'status': check_status_data.get('status', 'running'),
        'progress': check_status_data.get('progress', 0),
        'total_accounts': check_status_data.get('total_accounts', 0),
        'completed_accounts': check_status_data.get('completed_accounts', 0),
        'success': check_status_data.get('success', 0),
        'failed': check_status_data.get('failed', 0),
        'start_time': check_status_data.get('start_time', datetime.now().isoformat()),
        'end_time': check_status_data.get('end_time'),
        'checking_ip': check_status_data.get('checking_ip'),
        'config': check_status_data.get('config'),  # Save config used for this check
        'service': check_status_data.get('service', 'Microsoft Account Checker'),  # Save service name
        'results': check_results_data
    })
    
    return save_user_data(username, user_info)


def restore_active_checks():
    """
    Restore active checks from user data
    Returns dict of restored checks: {check_id: {status, ...}}
    """
    restored_checks = {}
    load_user_data()
    
    for username, user_info in user_data_cache.items():
        user_checks = user_info.get('checks', {})
        for check_id, check_info in user_checks.items():
            if check_info.get('status') == 'running':
                restored_checks[check_id] = {
                    'status': 'running',
                    'progress': check_info.get('progress', 0),
                    'total_accounts': check_info.get('total_accounts', 0),
                    'completed_accounts': check_info.get('completed_accounts', 0),
                    'success': check_info.get('success', 0),
                    'failed': check_info.get('failed', 0),
                    'start_time': check_info.get('start_time', datetime.now().isoformat()),
                    'username': username,
                    'checking_ip': check_info.get('checking_ip'),
                    'config': check_info.get('config')  # Restore config used for this check
                }
    
    return restored_checks


def generate_access_key(length=16):
    """Generate random access key"""
    import secrets
    import string
    alphabet = string.ascii_letters + string.digits
    return ''.join(secrets.choice(alphabet) for _ in range(length))


def generate_username(prefix="user"):
    """Generate unique username"""
    import secrets
    import string
    # Generate random suffix
    suffix = ''.join(secrets.choice(string.digits) for _ in range(6))
    username = f"{prefix}{suffix}"
    
    # Ensure uniqueness
    counter = 1
    while user_exists(username):
        username = f"{prefix}{suffix}{counter}"
        counter += 1
    
    return username


def generate_credentials_from_telegram(chat_id):
    """
    Generate Access Key and Username from Telegram Bot interaction using Chat ID
    
    Args:
        chat_id: Telegram Chat ID (required)
    
    Returns:
        dict: {'success': bool, 'username': str, 'access_key': str, 'message': str}
    """
    try:
        # Verify chat_id is provided
        if not chat_id:
            return {
                'success': False,
                'message': f'Chat ID is required. Please contact admin (Chat ID: {TELEGRAM_ADMIN_CHAT_ID})'
            }
        
        # Verify user exists in Telegram
        url = f"{TELEGRAM_API_URL}/getChat"
        params = {"chat_id": chat_id}
        response = requests.get(url, params=params, timeout=10)
        
        if response.status_code != 200 or not response.json().get('ok'):
            return {
                'success': False,
                'message': f'Invalid Telegram Chat ID. Please contact admin (Chat ID: {TELEGRAM_ADMIN_CHAT_ID})'
            }
        
        # Check if user already registered by chat_id
        load_user_data()
        for username, user_info in user_data_cache.items():
            if user_info.get('chat_id') == str(chat_id):
                return {
                    'success': False,
                    'message': 'You are already registered. Please use your existing credentials.'
                }
        
        # Generate credentials
        username = generate_username()
        access_key = generate_access_key()
        
        # Check if this is admin
        is_admin = (str(chat_id) == TELEGRAM_ADMIN_CHAT_ID)
        
        # Create user data
        user_data = {
            'username': username,
            'key_hash': hash_key(access_key),
            'created_at': datetime.now().isoformat(),
            'last_used': datetime.now().isoformat(),
            'is_admin': is_admin,
            'chat_id': str(chat_id),
            'checks': {},
            'total_checks': 0,
            'config': {
                'use_proxy': False,
                'proxy_type': 'http',
                'proxy_host': 'proxyhost:port',
                'proxy_username': 'username',
                'proxy_password': 'password',
                'threads': 5,
                'list_file': os.path.join(ROOT_DIR, 'list.txt')
            }
        }
        
        # Save user data
        if not save_user_data(username, user_data):
            return {
                'success': False,
                'message': f'Failed to save user data. Please contact admin (Chat ID: {TELEGRAM_ADMIN_CHAT_ID})'
            }
        
        # Send credentials via Telegram
        message = f"""🎉 <b>Registration Successful!</b>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

✅ <b>Your account has been created!</b>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

👤 <b>Login Credentials:</b>

🔹 <b>Username:</b>
<code>{username}</code>

🔑 <b>Access Key:</b>
<code>{access_key}</code>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

⚠️ <b>IMPORTANT - SAVE THESE CREDENTIALS!</b>

🔒 Save credentials securely!
📝 You will need them to login.
🔐 After login, you will receive an OTP code via Telegram.

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

🌐 <b>Login Now:</b>
{get_base_url()}/login

🤖 <b>Bot:</b> @{TELEGRAM_BOT_USERNAME}

💬 <b>Support:</b>
Chat ID: {TELEGRAM_ADMIN_CHAT_ID}

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"""
        
        if send_telegram_message(chat_id, message):
            return {
                'success': True,
                'username': username,
                'access_key': access_key,
                'message': 'Credentials sent to your Telegram. Please check your messages.'
            }
        else:
            return {
                'success': False,
                'message': f'Failed to send credentials via Telegram. Please contact admin (Chat ID: {TELEGRAM_ADMIN_CHAT_ID})'
            }
            
    except Exception as e:
        print(f"Error generating credentials from Telegram: {e}")
        return {
            'success': False,
            'message': f'Error: {str(e)}. Please contact @pleasegiftmemoney'
        }


def generate_otp():
    """Generate 6-digit OTP"""
    import secrets
    return ''.join([str(secrets.randbelow(10)) for _ in range(6)])


def send_otp(username, chat_id=None):
    """
    Send OTP code to user via Telegram using Chat ID
    
    Args:
        username: Username
        chat_id: Telegram Chat ID (optional, will get from user data if not provided)
    
    Returns:
        dict: {'success': bool, 'otp': str, 'message': str}
    """
    try:
        # Get user data
        user_info = get_user(username)
        if not user_info:
            return {'success': False, 'message': 'User not found'}
        
        # Get Chat ID
        if not chat_id:
            chat_id = user_info.get('chat_id') or user_info.get('telegram_id')  # Support legacy telegram_id
        
        if not chat_id:
            return {'success': False, 'message': f'Chat ID not found. Please contact admin (Chat ID: {TELEGRAM_ADMIN_CHAT_ID})'}
        
        # Generate OTP
        otp = generate_otp()
        expires_at = datetime.now().timestamp() + 300  # 5 minutes expiry
        
        # Store OTP
        otp_storage[username] = {
            'otp': otp,
            'expires': expires_at,
            'attempts': 0
        }
        
        # Send OTP via Telegram
        message = f"""🔐 <b>OTP Verification Code</b>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

👤 <b>Account:</b> <code>{username}</code>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

🔑 <b>Your OTP Code:</b>

<code>{otp}</code>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

⏰ <b>Validity:</b> 5 minutes
🔒 <b>Security:</b> Do not share this code with anyone

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

⚠️ <b>Security Alert:</b>
If you didn't request this code, please contact admin immediately.

💬 <b>Support:</b>
Chat ID: {TELEGRAM_ADMIN_CHAT_ID}

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"""
        
        if send_telegram_message(chat_id, message):
            return {
                'success': True,
                'otp': otp,  # For testing, remove in production
                'message': 'OTP sent to your Telegram'
            }
        else:
            return {'success': False, 'message': 'Failed to send OTP via Telegram'}
            
    except Exception as e:
        print(f"Error sending OTP: {e}")
        return {'success': False, 'message': f'Error: {str(e)}'}


def verify_otp(username, otp_code):
    """
    Verify OTP code
    
    Args:
        username: Username
        otp_code: OTP code entered by user
    
    Returns:
        dict: {'success': bool, 'message': str}
    """
    if username not in otp_storage:
        return {'success': False, 'message': 'OTP not found. Please request a new OTP.'}
    
    otp_data = otp_storage[username]
    
    # Check expiry
    if datetime.now().timestamp() > otp_data['expires']:
        del otp_storage[username]
        return {'success': False, 'message': 'OTP has expired. Please request a new OTP.'}
    
    # Check attempts
    if otp_data['attempts'] >= 3:
        del otp_storage[username]
        return {'success': False, 'message': 'Too many failed attempts. Please request a new OTP.'}
    
    # Verify OTP
    if otp_data['otp'] == otp_code:
        # OTP verified, remove from storage
        del otp_storage[username]
        return {'success': True, 'message': 'OTP verified successfully'}
    else:
        otp_data['attempts'] += 1
        return {'success': False, 'message': f'Invalid OTP. {3 - otp_data["attempts"]} attempts remaining.'}


def request_otp(username):
    """
    Request OTP for login
    
    Args:
        username: Username
    
    Returns:
        dict: {'success': bool, 'message': str}
    """
    return send_otp(username)


# Initialize user data on import
load_user_data()

