"""
Run Python API Server and Telegram Bot Together
This script starts both Flask API server and Telegram Bot simultaneously

Usage:
    python run.py

Or with custom port:
    API_PORT=5001 python run.py
"""
import os
import sys
import threading
import time
import signal

# Add current directory to path
sys.path.insert(0, os.path.dirname(os.path.abspath(__file__)))

# Global flag for graceful shutdown
shutdown_flag = threading.Event()

def run_api_server():
    """Run Flask API Server in a separate thread"""
    try:
        import func
        # Load user data on startup
        func.load_user_data()
        
        from api_server import app
        
        # Get port from environment or default to 5001
        port = int(os.getenv('API_PORT', '5001'))
        host = os.getenv('API_HOST', '0.0.0.0')  # Allow external connections
        
        print(f"\n{'='*60}")
        print(f"[API SERVER] Starting Python API Server...")
        print(f"{'='*60}")
        print(f"[API SERVER] Host: {host}")
        print(f"[API SERVER] Port: {port}")
        print(f"[API SERVER] API URL: http://{host if host != '0.0.0.0' else 'localhost'}:{port}")
        print(f"{'='*60}\n")
        
        # Run Flask app (blocking call)
        app.run(host=host, port=port, debug=False, use_reloader=False, threaded=True)
    except Exception as e:
        print(f"[API SERVER] Error: {e}")
        import traceback
        traceback.print_exc()
        shutdown_flag.set()

def run_telegram_bot():
    """Run Telegram Bot in main thread"""
    try:
        print(f"\n{'='*60}")
        print(f"[TELEGRAM BOT] Starting Telegram Bot...")
        print(f"{'='*60}\n")
        
        from telegram_bot import main as bot_main
        bot_main()
    except Exception as e:
        print(f"[TELEGRAM BOT] Error: {e}")
        import traceback
        traceback.print_exc()
        shutdown_flag.set()

def signal_handler(sig, frame):
    """Handle shutdown signals (Ctrl+C)"""
    print("\n\n[SHUTDOWN] Received shutdown signal (Ctrl+C). Stopping services...")
    shutdown_flag.set()
    print("[SHUTDOWN] Please wait for services to stop gracefully...")
    time.sleep(1)
    sys.exit(0)

def main():
    """Main function to run both services"""
    # Register signal handlers for graceful shutdown
    if sys.platform != 'win32':
        signal.signal(signal.SIGINT, signal_handler)
        signal.signal(signal.SIGTERM, signal_handler)
    
    print(f"\n{'='*80}")
    print(f"🚀 Starting 403 Forbidden Services")
    print(f"{'='*80}")
    print(f"Services to start:")
    print(f"  1. ✅ Python API Server (Flask)")
    print(f"  2. ✅ Telegram Bot")
    print(f"{'='*80}\n")
    
    # Start API Server in a separate thread (daemon thread)
    api_thread = threading.Thread(target=run_api_server, daemon=True, name="API-Server-Thread")
    api_thread.start()
    
    # Wait a bit for API server to initialize
    print("[MAIN] Waiting for API server to initialize...")
    time.sleep(3)
    
    # Check if API server thread is still alive
    if not api_thread.is_alive():
        print("[ERROR] API Server thread died. Check for errors above.")
        sys.exit(1)
    
    print("[MAIN] API Server started successfully!")
    print("[MAIN] Starting Telegram Bot...\n")
    
    # Start Telegram Bot in main thread (blocking)
    try:
        run_telegram_bot()
    except KeyboardInterrupt:
        print("\n[SHUTDOWN] Keyboard interrupt received. Stopping services...")
    except Exception as e:
        print(f"\n[ERROR] Fatal error: {e}")
        import traceback
        traceback.print_exc()
    finally:
        shutdown_flag.set()
        print("\n[SHUTDOWN] All services stopped.")
        print("[SHUTDOWN] Goodbye!")

if __name__ == '__main__':
    main()
