"""
Telegram Bot Module for Microsoft Account Checker
Centralized Telegram Bot functionality - used by func.py and microsoft.py

Requirements:
    pip install python-telegram-bot

Run bot standalone:
    python telegram_bot.py
"""
import os
import json
import requests
from datetime import datetime

try:
    from telegram import Update, InlineKeyboardButton, InlineKeyboardMarkup, KeyboardButton, ReplyKeyboardMarkup, ReplyKeyboardRemove, WebAppInfo
    from telegram.ext import ApplicationBuilder, CommandHandler, ContextTypes, ConversationHandler, MessageHandler, filters
    TELEGRAM_LIB_AVAILABLE = True
except ImportError:
    TELEGRAM_LIB_AVAILABLE = False
    print("Warning: python-telegram-bot library not installed. Bot handlers will not work.")
    print("Install with: pip install python-telegram-bot")

# Telegram Bot Configuration
BOT_TOKEN = "8283589446:AAFcoaT-VMrQiFutqRVA2Ty-Sr0P5bxB990"
TELEGRAM_API_URL = f"https://api.telegram.org/bot{BOT_TOKEN}"
TELEGRAM_ADMIN_CHAT_ID = "1859095647"
TELEGRAM_BOT_USERNAME = "forbidden_tools_bot"
REQUIRED_CHANNEL = "c_forbidden"  # Required channel to join (username without @)

# User data storage - will use func.USERS_FOLDER when needed
# Don't import func here to avoid circular import
# USERS_FOLDER will be accessed via func.USERS_FOLDER in functions that need it
USERS_FOLDER = None  # Will be set dynamically when func is imported

# Conversation states for registration
(REG_NAME, REG_PHONE, REG_CHANNEL) = range(3)

# Base URL helper
def get_base_url():
    """Get base URL (http or https) based on configuration"""
    # Note: This is mainly for Telegram bot messages, PHP web interface doesn't use this
    use_https = os.getenv('USE_HTTPS', 'false').lower() == 'true'
    if use_https:
        port = int(os.getenv('HTTPS_PORT', '443'))
        if port == 443:
            return "https://localhost"
        return f"https://localhost:{port}"
    else:
        port = int(os.getenv('HTTP_PORT', '5000'))
        if port == 80:
            return "http://localhost"
        return f"http://localhost:{port}"

# Debug initialization
_init_timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
print(f"[TELEGRAM_BOT] [{_init_timestamp}] Module loaded")
print(f"[TELEGRAM_BOT] Bot Username: @{TELEGRAM_BOT_USERNAME}")
print(f"[TELEGRAM_BOT] Admin Chat ID: {TELEGRAM_ADMIN_CHAT_ID}")
print(f"[TELEGRAM_BOT] Telegram library available: {TELEGRAM_LIB_AVAILABLE}")


def send_telegram_message(chat_id, message):
    """
    Send message via Telegram Bot API
    Used by func.py and microsoft.py
    """
    timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    print(f"\n{'='*80}")
    print(f"[TELEGRAM DEBUG] [{timestamp}] Sending message to Telegram")
    print(f"[TELEGRAM DEBUG] Chat ID: {chat_id}")
    print(f"[TELEGRAM DEBUG] Message length: {len(message)} characters")
    print(f"[TELEGRAM DEBUG] Message preview: {message[:100]}..." if len(message) > 100 else f"[TELEGRAM DEBUG] Message: {message}")
    
    try:
        url = f"{TELEGRAM_API_URL}/sendMessage"
        params = {
            "chat_id": chat_id,
            "text": message,
            "parse_mode": "HTML"
        }
        
        print(f"[TELEGRAM DEBUG] Request URL: {url}")
        print(f"[TELEGRAM DEBUG] Request params: chat_id={chat_id}, parse_mode=HTML")
        
        response = requests.post(url, params=params, timeout=10)
        result = response.json()
        
        print(f"[TELEGRAM DEBUG] Response status code: {response.status_code}")
        print(f"[TELEGRAM DEBUG] Response JSON: {json.dumps(result, indent=2)}")
        
        if response.status_code == 200 and result.get('ok'):
            message_id = result.get('result', {}).get('message_id', 'N/A')
            print(f"[TELEGRAM DEBUG] ✓ Successfully sent message to chat_id: {chat_id}, message_id: {message_id}")
            print(f"{'='*80}\n")
            return True
        else:
            error_code = result.get('error_code', 'N/A')
            description = result.get('description', 'N/A')
            print(f"[TELEGRAM DEBUG] ✗ Failed to send message to chat_id: {chat_id}")
            print(f"[TELEGRAM DEBUG] Error code: {error_code}, Description: {description}")
            print(f"{'='*80}\n")
            return False
    except Exception as e:
        print(f"[TELEGRAM DEBUG] ✗ Exception occurred: {type(e).__name__}: {str(e)}")
        import traceback
        traceback.print_exc()
        print(f"{'='*80}\n")
        return False


def check_channel_membership(user_id, channel_username):
    """
    Check if user is member of a Telegram channel
    Returns: True if member, False otherwise
    """
    try:
        # Add @ if not present
        channel_id = channel_username if channel_username.startswith('@') else f"@{channel_username}"
        url = f"{TELEGRAM_API_URL}/getChatMember"
        params = {
            "chat_id": channel_id,
            "user_id": user_id
        }
        response = requests.get(url, params=params, timeout=10)
        
        if response.status_code == 200:
            data = response.json()
            if data.get('ok'):
                status = data.get('result', {}).get('status', '')
                # Status can be: 'creator', 'administrator', 'member', 'restricted', 'left', 'kicked'
                return status in ['creator', 'administrator', 'member', 'restricted']
        return False
    except Exception as e:
        print(f"[TELEGRAM] Error checking channel membership: {e}")
        return False


def verify_telegram_user(chat_id, username=None):
    """
    Verify Telegram user with bot
    Used by func.py
    """
    timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    print(f"\n{'='*80}")
    print(f"[TELEGRAM DEBUG] [{timestamp}] Verifying Telegram user")
    print(f"[TELEGRAM DEBUG] Chat ID: {chat_id}")
    print(f"[TELEGRAM DEBUG] Username to verify: {username if username else 'Not provided'}")
    
    try:
        # Get chat information
        url = f"{TELEGRAM_API_URL}/getChat"
        params = {"chat_id": chat_id}
        
        print(f"[TELEGRAM DEBUG] Request URL: {url}")
        print(f"[TELEGRAM DEBUG] Request params: chat_id={chat_id}")
        
        response = requests.get(url, params=params, timeout=10)
        
        print(f"[TELEGRAM DEBUG] Response status code: {response.status_code}")
        
        if response.status_code == 200:
            data = response.json()
            print(f"[TELEGRAM DEBUG] Response JSON: {json.dumps(data, indent=2)}")
            
            if data.get('ok'):
                chat = data.get('result', {})
                chat_username = chat.get('username', 'N/A')
                first_name = chat.get('first_name', 'N/A')
                last_name = chat.get('last_name', 'N/A')
                chat_type = chat.get('type', 'N/A')
                
                print(f"[TELEGRAM DEBUG] Chat info:")
                print(f"[TELEGRAM DEBUG]   - Type: {chat_type}")
                print(f"[TELEGRAM DEBUG]   - Username: {chat_username}")
                print(f"[TELEGRAM DEBUG]   - First Name: {first_name}")
                print(f"[TELEGRAM DEBUG]   - Last Name: {last_name}")
                
                # Verify the username matches if provided
                if username:
                    if chat.get('username', '').lower() == username.lower():
                        print(f"[TELEGRAM DEBUG] ✓ Username match verified")
                        print(f"{'='*80}\n")
                        return True
                    else:
                        print(f"[TELEGRAM DEBUG] ✗ Username mismatch")
                # Or verify by first_name/last_name if username not available
                if chat.get('first_name') or chat.get('last_name'):
                    print(f"[TELEGRAM DEBUG] ✓ User verified by name")
                    print(f"{'='*80}\n")
                    return True
                
                print(f"[TELEGRAM DEBUG] ✗ Verification failed")
                print(f"{'='*80}\n")
                return False
            else:
                print(f"[TELEGRAM DEBUG] ✗ API returned error: {data.get('description', 'Unknown error')}")
                print(f"{'='*80}\n")
                return False
        else:
            print(f"[TELEGRAM DEBUG] ✗ HTTP error: {response.status_code}")
            print(f"{'='*80}\n")
            return False
    except Exception as e:
        print(f"[TELEGRAM DEBUG] ✗ Exception occurred: {type(e).__name__}: {str(e)}")
        import traceback
        traceback.print_exc()
        print(f"{'='*80}\n")
        return False


# Bot Handlers (only if library is available)
if TELEGRAM_LIB_AVAILABLE:
    async def retrieve_account(update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Handle /retrieve command - Get full account details"""
        chat_id = update.effective_chat.id
        
        try:
            import func
            func.load_user_data()
            
            # Find user by chat_id
            user_info = None
            username = None
            for uname, uinfo in func.user_data_cache.items():
                if uinfo.get('chat_id') == str(chat_id):
                    user_info = uinfo
                    username = uname
                    break
            
            if not user_info:
                await update.message.reply_text(
                    "❌ <b>Account not found!</b>\n\n"
                    "You are not registered yet. Please use /register to create an account.",
                    parse_mode='HTML'
                )
                return
            
            # Get access key (we need to find it from key_hash - we can't reverse hash, so we'll show a message)
            # Actually, we can't retrieve the original access key from hash, so we'll inform user
            name = user_info.get('name', 'N/A')
            phone = user_info.get('phone', 'N/A')
            telegram_username = user_info.get('telegram_username', 'N/A')
            saved_chat_id = user_info.get('chat_id', 'N/A')
            ip_address = user_info.get('ip_address', 'N/A')
            created_at = user_info.get('created_at', 'N/A')
            last_used = user_info.get('last_used', 'N/A')
            is_admin = user_info.get('is_admin', False)
            total_checks = user_info.get('total_checks', 0)
            
            # Format Telegram username display
            telegram_username_display = f"@{telegram_username}" if telegram_username and telegram_username != 'N/A' else "Not set"
            
            # Format account details
            account_details = f"""📋 <b>Account Details</b>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

👤 <b>Account Information:</b>

📝 <b>Name:</b>
<code>{name}</code>

👤 <b>Username:</b>
<code>{username}</code>

📱 <b>Telegram Username:</b>
<code>{telegram_username_display}</code>

📞 <b>Phone:</b>
<code>{phone}</code>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

🔐 <b>Security Information:</b>

🆔 <b>Chat ID:</b>
<code>{saved_chat_id}</code>

🌐 <b>IP Address:</b>
<code>{ip_address}</code>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

📊 <b>Account Statistics:</b>

👑 <b>Admin:</b>
{'✅ Yes' if is_admin else '❌ No'}

📈 <b>Total Checks:</b>
<code>{total_checks}</code>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

📅 <b>Account Dates:</b>

🆕 <b>Created:</b>
<code>{created_at}</code>

🔄 <b>Last Used:</b>
<code>{last_used}</code>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

⚠️ <b>Note:</b>
Access Key cannot be retrieved for security reasons.
If you forgot your access key, please contact admin.

🌐 <b>Login:</b>
{get_base_url()}/login

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"""
            
            # Create Mini App button
            mini_app_url = f"{get_base_url()}/miniapp.php"
            keyboard = [
                [KeyboardButton("📱 Open Mini App", web_app=WebAppInfo(url=mini_app_url))]
            ]
            reply_markup = ReplyKeyboardMarkup(keyboard, resize_keyboard=True)
            
            await update.message.reply_text(account_details, parse_mode='HTML', reply_markup=reply_markup)
            
        except Exception as e:
            print(f"[BOT] Error retrieving account: {e}")
            import traceback
            traceback.print_exc()
            await update.message.reply_text(
                f"❌ <b>Error retrieving account!</b>\n\n"
                f"Error: {str(e)}\n\n"
                "Please contact admin for assistance.",
                parse_mode='HTML'
            )
    
    async def start(update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Handle /start command - If registered, show account details. If not, start registration"""
        chat_id = update.effective_chat.id
        user_id = update.effective_user.id if update.effective_user else None
        
        # Check if user already registered
        try:
            import func
            func.load_user_data()
            for username, user_info in func.user_data_cache.items():
                if user_info.get('chat_id') == str(chat_id):
                    # User is registered, show account details
                    await retrieve_account(update, context)
                    return ConversationHandler.END
        except Exception as e:
            print(f"[BOT] Error checking existing user: {e}")
        
        # User not registered, start registration process
        # Continue with registration flow below
        
        # Show welcome message with Mini App button (if user might be registered but chat_id mismatch)
        welcome_msg = """🎉 <b>Welcome to 403 Forbidden!</b>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

📝 <b>New Account Registration</b>

To create an account, please use /register command.

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

💡 <b>Already registered?</b>
Use /retrieve to view your account details.

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"""
        
        await update.message.reply_text(welcome_msg, parse_mode='HTML')
        return ConversationHandler.END
        
    async def register_command(update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Handle /register command - Start registration process"""
        chat_id = update.effective_chat.id
        user_id = update.effective_user.id if update.effective_user else None
        
        # Check if user already registered
        try:
            import func
            func.load_user_data()
            for username, user_info in func.user_data_cache.items():
                if user_info.get('chat_id') == str(chat_id):
                    await update.message.reply_text(
                        "✅ <b>You are already registered!</b>\n\n"
                        "Use /retrieve to view your account details or /start to see your account information.",
                        parse_mode='HTML'
                    )
                    return ConversationHandler.END
        except Exception as e:
            print(f"[BOT] Error checking existing user: {e}")
        
        # Generate username from Telegram username (according to Telegram username)
        telegram_username = update.effective_user.username if update.effective_user and update.effective_user.username else None
        first_name = update.effective_user.first_name if update.effective_user else "User"
        last_name = update.effective_user.last_name if update.effective_user else ""
        
        # Generate username according to Telegram username
        import func
        import re
        
        if telegram_username:
            # Use Telegram username directly (lowercase, replace space with underscore)
            base_username = telegram_username.lower().replace(' ', '_')
            # Remove non-alphanumeric characters except underscore
            base_username = re.sub(r'[^a-z0-9_]', '', base_username)
            if len(base_username) < 3:
                # If too short, use combination of first_name
                base_username = first_name.lower().replace(' ', '_')
                base_username = re.sub(r'[^a-z0-9_]', '', base_username)
                if len(base_username) < 3:
                    base_username = "user"
        else:
            # If no Telegram username, use first_name
            base_username = first_name.lower().replace(' ', '_')
            base_username = re.sub(r'[^a-z0-9_]', '', base_username)
            if len(base_username) < 3:
                base_username = "user"
        
        # Use Telegram username directly (without random suffix)
        generated_username = base_username
        
        # Ensure username is unique - if already exists, add counter
        counter = 1
        while func.user_exists(generated_username):
            generated_username = f"{base_username}{counter}"
            counter += 1
        
        # Save generated username and Telegram username
        context.user_data['username'] = generated_username
        context.user_data['telegram_username'] = telegram_username  # Save Telegram username
        
        # Use first_name + last_name as default name
        default_name = f"{first_name} {last_name}".strip()
        if default_name:
            context.user_data['name'] = default_name
        
        # Start registration flow with buttons
        welcome_msg = f"""🎉 <b>Welcome to 403 Forbidden!</b>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

📝 <b>New Account Registration</b>

To create an account, please use the buttons below:
1️⃣ Send your full name (use button - mandatory)
2️⃣ Send your phone number (use Share Contact button - mandatory)
3️⃣ Verify mandatory channel join

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

✅ <b>Auto-Generated Username:</b>
<code>{generated_username}</code>

💡 Username is automatically generated from your Telegram username.

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

⚠️ <b>Important:</b> You must use the buttons below. Manual typing is not allowed.

💬 <b>Step 1: Send your full name</b>
(Click the button below)"""
        
        # Create keyboard with buttons - buttons only, no manual input
        keyboard = [
            [KeyboardButton("📝 Send Full Name", request_contact=True)],
            [KeyboardButton("📱 Share Phone Number", request_contact=True)]
        ]
        reply_markup = ReplyKeyboardMarkup(keyboard, resize_keyboard=True, one_time_keyboard=True)
        
        await update.message.reply_text(welcome_msg, parse_mode='HTML', reply_markup=reply_markup)
        return REG_NAME
    
    async def register_name(update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Handle name input - ONLY from contact button, no manual typing"""
        # Only accept contact sharing, reject manual text input
        if not update.message.contact:
            # Reject manual typing
            await update.message.reply_text(
                "❌ <b>Please use the button!</b>\n\n"
                "⚠️ Manual typing is not allowed.\n\n"
                "Please click the button below to send your information:",
                parse_mode='HTML',
                reply_markup=ReplyKeyboardMarkup([
                    [KeyboardButton("📝 Send Full Name", request_contact=True)],
                    [KeyboardButton("📱 Share Phone Number", request_contact=True)]
                ], resize_keyboard=True, one_time_keyboard=True)
            )
            return REG_NAME
        
        # User sent contact - extract name and phone
        contact = update.message.contact
        phone = contact.phone_number if contact.phone_number else None
        
        # Extract name from contact
        contact_name = contact.first_name or ""
        if contact.last_name:
            contact_name += f" {contact.last_name}"
        contact_name = contact_name.strip()
        
        # If no name in contact, use Telegram user name
        if not contact_name:
            first_name = update.effective_user.first_name if update.effective_user else ""
            last_name = update.effective_user.last_name if update.effective_user else ""
            contact_name = f"{first_name} {last_name}".strip()
        
        if not contact_name:
            contact_name = "User"
        
        # In REG_NAME state, we're collecting the name
        # Save the name
        context.user_data['name'] = contact_name
        
        # If phone also provided in the contact, save it too
        if phone:
            context.user_data['phone'] = phone
        
        # Check if we have both name and phone
        if 'name' in context.user_data and 'phone' in context.user_data:
            # Both collected, proceed to channel check
            await update.message.reply_text(
                f"✅ <b>Information received!</b>\n\n"
                f"📝 <b>Name:</b> <code>{contact_name}</code>\n"
                f"📱 <b>Phone:</b> <code>{phone}</code>\n\n"
                "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n"
                "⏳ <b>Verifying channel membership...</b>",
                parse_mode='HTML',
                reply_markup=ReplyKeyboardRemove()
            )
            return await proceed_to_channel_check(update, context)
        else:
            # Only name collected, ask for phone
            await update.message.reply_text(
                f"✅ <b>Full name received!</b>\n\n"
                f"📝 <b>Name:</b> <code>{contact_name}</code>\n\n"
                "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n"
                "📱 <b>Step 2: Send your phone number</b>\n"
                "⚠️ Please use the button below (manual typing not allowed):",
                parse_mode='HTML',
                reply_markup=ReplyKeyboardMarkup([
                    [KeyboardButton("📱 Share Phone Number", request_contact=True)]
                ], resize_keyboard=True, one_time_keyboard=True)
            )
            return REG_PHONE
    
    async def register_phone(update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Handle phone number input - ONLY from contact button, no manual typing"""
        # Only accept contact sharing, reject manual text input
        if not update.message.contact:
            # Reject manual typing
            await update.message.reply_text(
                "❌ <b>Please use the button!</b>\n\n"
                "⚠️ Manual typing is not allowed.\n\n"
                "Please click the button below to share your phone number:",
                parse_mode='HTML',
                reply_markup=ReplyKeyboardMarkup([
                    [KeyboardButton("📱 Share Phone Number", request_contact=True)]
                ], resize_keyboard=True, one_time_keyboard=True)
            )
            return REG_PHONE
        
        # User sent contact - extract phone number
        contact = update.message.contact
        phone = contact.phone_number if contact.phone_number else None
        
        if not phone:
            await update.message.reply_text(
                "❌ <b>Phone number not found in contact!</b>\n\n"
                "Please try again using the button:",
                parse_mode='HTML',
                reply_markup=ReplyKeyboardMarkup([
                    [KeyboardButton("📱 Share Phone Number", request_contact=True)]
                ], resize_keyboard=True, one_time_keyboard=True)
            )
            return REG_PHONE
        
        # Save phone number
        context.user_data['phone'] = phone
        
        # If name not set, try to get from contact
        if 'name' not in context.user_data or not context.user_data.get('name'):
            contact_name = contact.first_name or ""
            if contact.last_name:
                contact_name += f" {contact.last_name}"
            contact_name = contact_name.strip()
            
            if not contact_name:
                first_name = update.effective_user.first_name if update.effective_user else ""
                last_name = update.effective_user.last_name if update.effective_user else ""
                contact_name = f"{first_name} {last_name}".strip()
            
            if not contact_name:
                contact_name = "User"
            
            context.user_data['name'] = contact_name
        
        # Both name and phone collected, proceed to channel check
        return await proceed_to_channel_check(update, context)
    
    async def proceed_to_channel_check(update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Proceed to channel membership check"""
        # Remove keyboard
        await update.message.reply_text(
            "✅ <b>Data received!</b>\n\n"
            "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n"
            "⏳ <b>Verifying channel membership...</b>",
            parse_mode='HTML',
            reply_markup=ReplyKeyboardRemove()
        )
        
        # Check channel membership
        user_id = update.effective_user.id if update.effective_user else update.effective_chat.id
        is_member = check_channel_membership(user_id, REQUIRED_CHANNEL)
        
        if is_member:
            # User is member, proceed with registration
            return await complete_registration(update, context)
        else:
            # User not member, ask to join
            keyboard = InlineKeyboardMarkup([[
                InlineKeyboardButton("🔗 Join Channel", url=f"https://t.me/{REQUIRED_CHANNEL}")
            ], [
                InlineKeyboardButton("✅ Already Joined", callback_data="check_channel")
            ]])
            
            await update.message.reply_text(
                "⚠️ <b>Mandatory Channel Join Required!</b>\n\n"
                "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n"
                "📢 You must join the following channel to continue registration:\n\n"
                f"🔗 <b>@{REQUIRED_CHANNEL}</b>\n\n"
                "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n"
                "1️⃣ Click the button below to join the channel\n"
                "2️⃣ After joining, click '✅ Already Joined' button\n\n"
                "⚠️ <b>Important:</b> Bot must be added to the channel as admin for verification.",
                parse_mode='HTML',
                reply_markup=keyboard
            )
            return REG_CHANNEL
    
    
    async def check_channel_button(update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Handle channel check button"""
        query = update.callback_query
        
        if not query:
            print("[BOT] Error: No callback query in check_channel_button")
            return REG_CHANNEL
        
        # Always answer callback query first to remove loading state
        try:
            await query.answer(text="Checking channel membership...", show_alert=False)
        except Exception as e:
            print(f"[BOT] Error answering callback query: {e}")
        
        # Get user_id from callback query
        user_id = query.from_user.id if query.from_user else update.effective_user.id
        is_member = check_channel_membership(user_id, REQUIRED_CHANNEL)
        
        if is_member:
            # User is member, proceed with registration
            try:
                await query.answer(text="✅ Channel membership verified!", show_alert=False)
            except:
                pass
            return await complete_registration(update, context)
        else:
            # User not member, ask to join
            keyboard = InlineKeyboardMarkup([[
                InlineKeyboardButton("🔗 Join Channel", url=f"https://t.me/{REQUIRED_CHANNEL}")
            ], [
                InlineKeyboardButton("✅ Already Joined", callback_data="check_channel")
            ]])
            
            try:
                await query.edit_message_text(
                    "❌ <b>You haven't joined the channel yet!</b>\n\n"
                    "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n"
                    "Please join the channel first:\n\n"
                    f"🔗 <b>@{REQUIRED_CHANNEL}</b>\n\n"
                    "After joining, click '✅ Already Joined' button again.",
                    parse_mode='HTML',
                    reply_markup=keyboard
                )
            except Exception as e:
                print(f"[BOT] Error editing message: {e}")
                # Try to send new message if edit fails
                try:
                    await query.message.reply_text(
                        "❌ <b>You haven't joined the channel yet!</b>\n\n"
                        "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n"
                        "Please join the channel first:\n\n"
                        f"🔗 <b>@{REQUIRED_CHANNEL}</b>\n\n"
                        "After joining, click '✅ Already Joined' button again.",
                        parse_mode='HTML',
                        reply_markup=keyboard
                    )
                except Exception as e2:
                    print(f"[BOT] Error sending message: {e2}")
            
            return REG_CHANNEL
    
    async def complete_registration(update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Complete registration process"""
        try:
            import func
            
            name = context.user_data.get('name')
            username = context.user_data.get('username')
            telegram_username = context.user_data.get('telegram_username')
            phone = context.user_data.get('phone')
            chat_id = update.effective_chat.id
            
            # Generate access key
            access_key = func.generate_access_key()
            
            # Create user data
            user_data = {
                'username': username,
                'name': name,
                'phone': phone,
            'telegram_username': telegram_username,  # Save Telegram username
            'key_hash': func.hash_key(access_key),
            'created_at': datetime.now().isoformat(),
            'last_used': datetime.now().isoformat(),
            'is_admin': (str(chat_id) == TELEGRAM_ADMIN_CHAT_ID),
            'chat_id': str(chat_id),  # Save Chat ID
            'ip_address': 'Unknown',  # IP Address (will be updated from web interface if available)
            'telegram_notification': False,  # Default: notification disabled
                'checks': {},
                'total_checks': 0,
                'config': {
                    'use_proxy': False,
                    'proxy_type': 'http',
                    'proxy_host': 'proxyhost:port',
                    'proxy_username': 'username',
                    'proxy_password': 'password',
                    'threads': 5,
                    'list_file': os.path.join(os.path.dirname(os.path.dirname(os.path.abspath(__file__))), 'list.txt')
                }
            }
            
            # Save user data
            if func.save_user_data(username, user_data):
                # Send success message
                telegram_username_display = f"@{telegram_username}" if telegram_username else "Not set"
                success_msg = f"""🎉 <b>Registration Successful!</b>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

✅ <b>Your account has been created successfully!</b>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

👤 <b>Account Information:</b>

📝 <b>Name:</b>
<code>{name}</code>

👤 <b>Username:</b>
<code>{username}</code>

📱 <b>Telegram Username:</b>
<code>{telegram_username_display}</code>

📞 <b>Phone:</b>
<code>{phone}</code>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

🔑 <b>Login Credentials:</b>

👤 <b>Username:</b>
<code>{username}</code>

🔑 <b>Access Key:</b>
<code>{access_key}</code>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

⚠️ <b>IMPORTANT - SAVE THESE CREDENTIALS!</b>

🔒 Save credentials securely!
📝 You will need them to login.
🔐 After login, you will receive an OTP code via Telegram.

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

🌐 <b>Login Now:</b>
{get_base_url()}/login

🤖 <b>Bot:</b> @{TELEGRAM_BOT_USERNAME}

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"""
                
                if update.callback_query:
                    try:
                        await update.callback_query.answer()
                        await update.callback_query.edit_message_text(success_msg, parse_mode='HTML')
                    except Exception as e:
                        print(f"[BOT] Error handling callback query in success: {e}")
                        try:
                            await update.callback_query.message.reply_text(success_msg, parse_mode='HTML')
                        except:
                            pass
                elif update.message:
                    await update.message.reply_text(success_msg, parse_mode='HTML')
                else:
                    # Fallback: send new message
                    chat_id = update.effective_chat.id
                    send_telegram_message(chat_id, success_msg)
                
                # Clear user data
                context.user_data.clear()
                return ConversationHandler.END
            else:
                error_msg = "❌ <b>Failed to save data!</b>\n\nPlease contact admin for assistance."
                if update.callback_query:
                    try:
                        await update.callback_query.answer()
                        await update.callback_query.edit_message_text(error_msg, parse_mode='HTML')
                    except Exception as e:
                        print(f"[BOT] Error handling callback query in error: {e}")
                        try:
                            await update.callback_query.message.reply_text(error_msg, parse_mode='HTML')
                        except:
                            pass
                elif update.message:
                    await update.message.reply_text(error_msg, parse_mode='HTML')
                else:
                    chat_id = update.effective_chat.id
                    send_telegram_message(chat_id, error_msg)
                return ConversationHandler.END
                
        except Exception as e:
            print(f"[BOT] Error completing registration: {e}")
            import traceback
            traceback.print_exc()
            error_msg = f"❌ <b>An error occurred!</b>\n\nError: {str(e)}\n\nPlease contact admin."
            if update.callback_query:
                try:
                    await update.callback_query.answer()
                    await update.callback_query.edit_message_text(error_msg, parse_mode='HTML')
                except Exception as e2:
                    print(f"[BOT] Error handling callback query in exception: {e2}")
                    try:
                        await update.callback_query.message.reply_text(error_msg, parse_mode='HTML')
                    except:
                        pass
            elif update.message:
                await update.message.reply_text(error_msg, parse_mode='HTML')
            else:
                chat_id = update.effective_chat.id
                send_telegram_message(chat_id, error_msg)
            return ConversationHandler.END
    
    async def cancel(update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Cancel registration"""
        context.user_data.clear()
        await update.message.reply_text(
            "❌ <b>Registration cancelled.</b>\n\n"
            "Use /start to begin registration again.",
            parse_mode='HTML'
        )
        return ConversationHandler.END
    
    async def chatid(update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Handle /chatid command - Send user their Chat ID"""
        timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        chat_id = update.effective_chat.id
        chat_type = update.effective_chat.type
        user_id = update.effective_user.id if update.effective_user else 'N/A'
        username = update.effective_user.username if update.effective_user else 'N/A'
        first_name = update.effective_user.first_name if update.effective_user else 'N/A'
        
        print(f"\n{'='*80}")
        print(f"[BOT DEBUG] [{timestamp}] Received /chatid command")
        print(f"[BOT DEBUG] Chat ID: {chat_id}")
        print(f"[BOT DEBUG] Chat Type: {chat_type}")
        print(f"[BOT DEBUG] User ID: {user_id}")
        print(f"[BOT DEBUG] Username: @{username}" if username != 'N/A' else f"[BOT DEBUG] Username: {username}")
        print(f"[BOT DEBUG] First Name: {first_name}")
        print(f"[BOT DEBUG] Message ID: {update.message.message_id if update.message else 'N/A'}")
        
        # Format profesional dan rapi
        response_text = f"""🔍 <b>Your Telegram Chat ID</b>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

📱 <b>Chat ID:</b>
<code>{chat_id}</code>

📋 <b>Chat Type:</b>
<code>{chat_type}</code>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

💡 <b>How to use:</b>
Copy the Chat ID above and use it for registration on our website.

⚠️ <b>Important:</b>
Keep your Chat ID private and do not share it with others.

🔗 <b>Website:</b>
{get_base_url()}/register

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"""
        
        print(f"[BOT DEBUG] Sending response...")
        await update.message.reply_text(response_text, parse_mode='HTML')
        print(f"[BOT DEBUG] ✓ Response sent successfully")
        print(f"{'='*80}\n")
    
    def main():
        """Start the Telegram Bot with registration handlers"""
        timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        print(f"\n{'='*80}")
        print(f"[BOT DEBUG] [{timestamp}] Starting Telegram Bot")
        print(f"[BOT DEBUG] Bot Username: @{TELEGRAM_BOT_USERNAME}")
        print(f"[BOT DEBUG] Bot Token: {BOT_TOKEN[:10]}...{BOT_TOKEN[-10:]}")
        print(f"[BOT DEBUG] Admin Chat ID: {TELEGRAM_ADMIN_CHAT_ID}")
        print(f"[BOT DEBUG] Required Channel: {REQUIRED_CHANNEL}")
        print(f"[BOT DEBUG] API URL: {TELEGRAM_API_URL}")
        print(f"{'='*80}\n")
        
        app = ApplicationBuilder().token(BOT_TOKEN).build()
        
        # Registration conversation handler
        from telegram.ext import CallbackQueryHandler
        registration_handler = ConversationHandler(
            entry_points=[
                CommandHandler('start', start),
                CommandHandler('register', register_command)
            ],
            states={
                REG_NAME: [
                    MessageHandler(filters.TEXT & ~filters.COMMAND, register_name),
                    MessageHandler(filters.CONTACT, register_name)
                ],
                REG_PHONE: [
                    MessageHandler(filters.TEXT & ~filters.COMMAND, register_phone),
                    MessageHandler(filters.CONTACT, register_phone)
                ],
                REG_CHANNEL: [
                    CallbackQueryHandler(check_channel_button, pattern='^check_channel$')
                ],
            },
            fallbacks=[CommandHandler('cancel', cancel)]
        )
        
        # Add handlers
        # Admin command handlers
        async def admin_command(update: Update, context: ContextTypes.DEFAULT_TYPE):
            """Handle /admin command - Show admin menu"""
            chat_id = update.effective_chat.id
            
            try:
                import func
                func.load_user_data()
                
                # Check if user is admin
                user_info = None
                username = None
                for uname, uinfo in func.user_data_cache.items():
                    if uinfo.get('chat_id') == str(chat_id):
                        user_info = uinfo
                        username = uname
                        break
                
                if not user_info or not user_info.get('is_admin', False):
                    await update.message.reply_text(
                        "❌ <b>Access Denied!</b>\n\n"
                        "You must be an admin to use this command.",
                        parse_mode='HTML'
                    )
                    return
                
                admin_menu = f"""🔐 <b>Admin Panel</b>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

👤 <b>Welcome, {username}!</b>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

📋 <b>Available Commands:</b>

👥 <b>User Management:</b>
/adduser - Add new user
/deleteuser &lt;username&gt; - Delete user
/updateuser &lt;username&gt; - Update user
/userinfo &lt;username&gt; - Get full user info
/allusers - List all users

⚙️ <b>System Settings:</b>
/settings - View system settings

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

🌐 <b>Web Admin Panel:</b>
{get_base_url()}?page=admin

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"""
                await update.message.reply_text(admin_menu, parse_mode='HTML')
            except Exception as e:
                await update.message.reply_text(f"❌ Error: {str(e)}")
        
        async def all_users_command(update: Update, context: ContextTypes.DEFAULT_TYPE):
            """Handle /allusers command - List all users"""
            chat_id = update.effective_chat.id
            
            try:
                import func
                func.load_user_data()
                
                # Check if user is admin
                user_info = None
                for uname, uinfo in func.user_data_cache.items():
                    if uinfo.get('chat_id') == str(chat_id) and uinfo.get('is_admin', False):
                        user_info = uinfo
                        break
                
                if not user_info:
                    await update.message.reply_text("❌ Access Denied! Admin only.")
                    return
                
                users_list = "👥 <b>All Users</b>\n\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n"
                for uname, uinfo in func.user_data_cache.items():
                    role = "👑 Admin" if uinfo.get('is_admin', False) else "👤 User"
                    users_list += f"<code>{uname}</code> - {uinfo.get('name', 'N/A')} ({role})\n"
                
                users_list += "\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
                await update.message.reply_text(users_list, parse_mode='HTML')
            except Exception as e:
                await update.message.reply_text(f"❌ Error: {str(e)}")
        
        async def user_info_command(update: Update, context: ContextTypes.DEFAULT_TYPE):
            """Handle /userinfo command - Get full user info"""
            chat_id = update.effective_chat.id
            
            try:
                import func
                func.load_user_data()
                
                # Check if user is admin
                is_admin = False
                for uname, uinfo in func.user_data_cache.items():
                    if uinfo.get('chat_id') == str(chat_id) and uinfo.get('is_admin', False):
                        is_admin = True
                        break
                
                if not is_admin:
                    await update.message.reply_text("❌ Access Denied! Admin only.")
                    return
                
                target_username = ' '.join(context.args) if context.args else None
                if not target_username:
                    await update.message.reply_text("❌ Usage: /userinfo &lt;username&gt;")
                    return
                
                user_info = func.get_user(target_username)
                if not user_info:
                    await update.message.reply_text(f"❌ User '{target_username}' not found.")
                    return
                
                info = f"""📋 <b>User Information</b>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

👤 <b>Username:</b> <code>{target_username}</code>
📝 <b>Name:</b> {user_info.get('name', 'N/A')}
📞 <b>Phone:</b> {user_info.get('phone', 'N/A')}
📱 <b>Telegram:</b> {user_info.get('telegram_username', 'N/A')}
🆔 <b>Chat ID:</b> <code>{user_info.get('chat_id', 'N/A')}</code>
🌐 <b>IP Address:</b> <code>{user_info.get('ip_address', 'Unknown')}</code>
👑 <b>Role:</b> {'Admin' if user_info.get('is_admin', False) else 'User'}
🔔 <b>Notification:</b> {'Enabled' if user_info.get('telegram_notification', False) else 'Disabled'}
📊 <b>Total Checks:</b> {user_info.get('total_checks', 0)}
📅 <b>Created:</b> {user_info.get('created_at', 'N/A')}
🕐 <b>Last Used:</b> {user_info.get('last_used', 'N/A')}

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"""
                await update.message.reply_text(info, parse_mode='HTML')
            except Exception as e:
                await update.message.reply_text(f"❌ Error: {str(e)}")
        
        async def settings_command(update: Update, context: ContextTypes.DEFAULT_TYPE):
            """Handle /settings command - View system settings"""
            chat_id = update.effective_chat.id
            
            try:
                import func
                func.load_user_data()
                
                # Check if user is admin
                is_admin = False
                for uname, uinfo in func.user_data_cache.items():
                    if uinfo.get('chat_id') == str(chat_id) and uinfo.get('is_admin', False):
                        is_admin = True
                        break
                
                if not is_admin:
                    await update.message.reply_text("❌ Access Denied! Admin only.")
                    return
                
                settings_info = f"""⚙️ <b>System Settings</b>

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

🌐 <b>Base URL:</b>
<code>{get_base_url()}</code>

🤖 <b>Telegram Bot Token:</b>
<code>{BOT_TOKEN[:20]}...</code>

🆔 <b>Admin Chat ID:</b>
<code>{TELEGRAM_ADMIN_CHAT_ID}</code>

👤 <b>Bot Username:</b>
@{TELEGRAM_BOT_USERNAME}

📢 <b>Required Channel:</b>
@{REQUIRED_CHANNEL}

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

💡 <b>Note:</b> To change settings, please edit the configuration files.

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"""
                await update.message.reply_text(settings_info, parse_mode='HTML')
            except Exception as e:
                await update.message.reply_text(f"❌ Error: {str(e)}")
        
        async def add_user_command(update: Update, context: ContextTypes.DEFAULT_TYPE):
            """Handle /adduser command - Add new user (admin only)"""
            chat_id = update.effective_chat.id
            
            try:
                import func
                func.load_user_data()
                
                # Check if user is admin
                is_admin = False
                admin_username = None
                for uname, uinfo in func.user_data_cache.items():
                    if uinfo.get('chat_id') == str(chat_id) and uinfo.get('is_admin', False):
                        is_admin = True
                        admin_username = uname
                        break
                
                if not is_admin:
                    await update.message.reply_text("❌ Access Denied! Admin only.")
                    return
                
                args = context.args if context.args else []
                if len(args) < 2:
                    await update.message.reply_text(
                        "❌ <b>Usage:</b> /adduser &lt;username&gt; &lt;access_key&gt; [name] [phone] [telegram_username] [chat_id] [is_admin]\n\n"
                        "Example: /adduser testuser testkey123 John Doe +1234567890 @johndoe 123456789 0",
                        parse_mode='HTML'
                    )
                    return
                
                username = args[0]
                access_key = args[1]
                name = args[2] if len(args) > 2 else 'N/A'
                phone = args[3] if len(args) > 3 else 'N/A'
                telegram_username = args[4] if len(args) > 4 else 'N/A'
                user_chat_id = args[5] if len(args) > 5 else 'N/A'
                is_admin_user = args[6] == '1' if len(args) > 6 else False
                
                if func.user_exists(username):
                    await update.message.reply_text(f"❌ User '{username}' already exists!")
                    return
                
                # Create user
                from datetime import datetime
                user_data = {
                    'username': username,
                    'name': name,
                    'phone': phone,
                    'telegram_username': telegram_username.replace('@', '') if telegram_username != 'N/A' else 'N/A',
                    'key_hash': func.hash_key(access_key),
                    'created_at': datetime.now().isoformat(),
                    'last_used': datetime.now().isoformat(),
                    'is_admin': is_admin_user,
                    'chat_id': user_chat_id,
                    'ip_address': 'Unknown',
                    'telegram_notification': False,
                    'checks': {},
                    'total_checks': 0,
                    'config': {
                        'use_proxy': False,
                        'proxy_type': 'http',
                        'proxy_host': 'proxyhost:port',
                        'proxy_username': 'username',
                        'proxy_password': 'password',
                        'threads': 5,
                        'list_file': os.path.join(os.path.dirname(os.path.dirname(os.path.abspath(__file__))), 'list.txt')
                    }
                }
                
                if func.save_user_data(username, user_data):
                    await update.message.reply_text(
                        f"✅ <b>User Created Successfully!</b>\n\n"
                        f"👤 <b>Username:</b> <code>{username}</code>\n"
                        f"🔑 <b>Access Key:</b> <code>{access_key}</code>\n"
                        f"📝 <b>Name:</b> {name}\n"
                        f"👑 <b>Role:</b> {'Admin' if is_admin_user else 'User'}\n\n"
                        f"⚠️ <b>Save the access key securely!</b>",
                        parse_mode='HTML'
                    )
                else:
                    await update.message.reply_text("❌ Failed to create user.")
            except Exception as e:
                await update.message.reply_text(f"❌ Error: {str(e)}")
        
        async def delete_user_command(update: Update, context: ContextTypes.DEFAULT_TYPE):
            """Handle /deleteuser command - Delete user (admin only)"""
            chat_id = update.effective_chat.id
            
            try:
                import func
                func.load_user_data()
                
                # Check if user is admin
                is_admin = False
                admin_username = None
                for uname, uinfo in func.user_data_cache.items():
                    if uinfo.get('chat_id') == str(chat_id) and uinfo.get('is_admin', False):
                        is_admin = True
                        admin_username = uname
                        break
                
                if not is_admin:
                    await update.message.reply_text("❌ Access Denied! Admin only.")
                    return
                
                args = context.args if context.args else []
                if not args:
                    await update.message.reply_text("❌ Usage: /deleteuser &lt;username&gt;", parse_mode='HTML')
                    return
                
                target_username = args[0]
                
                if target_username == admin_username:
                    await update.message.reply_text("❌ Cannot delete your own account!")
                    return
                
                if not func.user_exists(target_username):
                    await update.message.reply_text(f"❌ User '{target_username}' not found!")
                    return
                
                import os
                filepath = os.path.join(func.USERS_FOLDER, f"{target_username}.json")
                if os.path.exists(filepath):
                    os.remove(filepath)
                    func.load_user_data()  # Reload cache
                    await update.message.reply_text(f"✅ User '{target_username}' deleted successfully!")
                else:
                    await update.message.reply_text(f"❌ User file not found!")
            except Exception as e:
                await update.message.reply_text(f"❌ Error: {str(e)}")
        
        async def update_user_command(update: Update, context: ContextTypes.DEFAULT_TYPE):
            """Handle /updateuser command - Update user (admin only)"""
            chat_id = update.effective_chat.id
            
            try:
                import func
                func.load_user_data()
                
                # Check if user is admin
                is_admin = False
                admin_username = None
                for uname, uinfo in func.user_data_cache.items():
                    if uinfo.get('chat_id') == str(chat_id) and uinfo.get('is_admin', False):
                        is_admin = True
                        admin_username = uname
                        break
                
                if not is_admin:
                    await update.message.reply_text("❌ Access Denied! Admin only.")
                    return
                
                args = context.args if context.args else []
                if len(args) < 2:
                    await update.message.reply_text(
                        "❌ <b>Usage:</b> /updateuser &lt;username&gt; &lt;field&gt; &lt;value&gt;\n\n"
                        "Fields: name, phone, telegram_username, chat_id, access_key, is_admin, telegram_notification\n\n"
                        "Example: /updateuser testuser name \"John Doe\"\n"
                        "Example: /updateuser testuser is_admin 1",
                        parse_mode='HTML'
                    )
                    return
                
                target_username = args[0]
                field = args[1]
                value = ' '.join(args[2:]) if len(args) > 2 else ''
                
                user_info = func.get_user(target_username)
                if not user_info:
                    await update.message.reply_text(f"❌ User '{target_username}' not found!")
                    return
                
                # Update field
                if field == 'name':
                    user_info['name'] = value
                elif field == 'phone':
                    user_info['phone'] = value
                elif field == 'telegram_username':
                    user_info['telegram_username'] = value.replace('@', '')
                elif field == 'chat_id':
                    user_info['chat_id'] = value
                elif field == 'access_key':
                    user_info['key_hash'] = func.hash_key(value)
                elif field == 'is_admin':
                    if target_username == admin_username and value != '1':
                        await update.message.reply_text("❌ Cannot remove admin status from your own account!")
                        return
                    user_info['is_admin'] = value == '1'
                elif field == 'telegram_notification':
                    user_info['telegram_notification'] = value == '1'
                else:
                    await update.message.reply_text(f"❌ Invalid field: {field}")
                    return
                
                if func.save_user_data(target_username, user_info):
                    await update.message.reply_text(f"✅ User '{target_username}' updated successfully!\n\nField: {field}\nNew value: {value}")
                else:
                    await update.message.reply_text("❌ Failed to update user.")
            except Exception as e:
                await update.message.reply_text(f"❌ Error: {str(e)}")
        
        # Add handlers - registration_handler must be added first to handle its callbacks
        # ConversationHandler should handle its own callback queries
        app.add_handler(registration_handler)
        
        # Add global callback query handler as fallback (only for unhandled callbacks)
        # This should be added AFTER ConversationHandler so it only catches unmatched callbacks
        async def global_callback_handler(update: Update, context: ContextTypes.DEFAULT_TYPE):
            """Global callback query handler to answer all callback queries that aren't handled"""
            query = update.callback_query
            if query:
                try:
                    # Answer callback query to remove loading state
                    # This will be called for any callback query that doesn't match other handlers
                    await query.answer(text="Processing...", show_alert=False)
                    print(f"[BOT] Unhandled callback query: {query.data}")
                except Exception as e:
                    print(f"[BOT] Error in global callback handler: {e}")
        
        # Register global callback query handler with high group number (low priority)
        # This ensures it only handles callbacks that other handlers don't catch
        # Don't use pattern to let ConversationHandler handle its own callbacks first
        from telegram.ext import CallbackQueryHandler as CQH
        app.add_handler(CQH(global_callback_handler), group=99)
        app.add_handler(CommandHandler("chatid", chatid))
        app.add_handler(CommandHandler("retrieve", retrieve_account))
        app.add_handler(CommandHandler("admin", admin_command))
        app.add_handler(CommandHandler("allusers", all_users_command))
        app.add_handler(CommandHandler("userinfo", user_info_command))
        app.add_handler(CommandHandler("settings", settings_command))
        app.add_handler(CommandHandler("adduser", add_user_command))
        app.add_handler(CommandHandler("deleteuser", delete_user_command))
        app.add_handler(CommandHandler("updateuser", update_user_command))
        
        print(f"[BOT DEBUG] Handler registered: Global Callback Query Handler (group=1)")
        print(f"[BOT DEBUG] Handler registered: /start (smart - shows account if registered, else register)")
        print(f"[BOT DEBUG] Handler registered: /register (registration)")
        print(f"[BOT DEBUG] Handler registered: /retrieve (account details)")
        print(f"[BOT DEBUG] Handler registered: /chatid")
        print(f"[BOT DEBUG] Handler registered: /admin (admin panel)")
        print(f"[BOT DEBUG] Handler registered: /allusers (list all users)")
        print(f"[BOT DEBUG] Handler registered: /userinfo (get user info)")
        print(f"[BOT DEBUG] Handler registered: /settings (system settings)")
        print(f"[BOT DEBUG] Handler registered: /adduser (add new user)")
        print(f"[BOT DEBUG] Handler registered: /deleteuser (delete user)")
        print(f"[BOT DEBUG] Handler registered: /updateuser (update user)")
        print(f"[BOT DEBUG] Bot is running with polling...")
        print(f"[BOT DEBUG] Commands available: /start, /register, /retrieve, /chatid, /admin, /allusers, /userinfo, /settings, /adduser, /deleteuser, /updateuser, /cancel")
        print(f"[BOT DEBUG] Waiting for updates...\n")
        
        # Run bot with polling
        app.run_polling()
else:
    def main():
        """Bot cannot run without python-telegram-bot library"""
        print("Error: python-telegram-bot library not installed.")
        print("Please install it with: pip install python-telegram-bot")
        exit(1)


if __name__ == "__main__":
    main()

